#############################################################################################
##### Parent-teacher discrepancy modelling for the pro-social behaviour scale sweep 8    ####
#############################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw8-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw8clean(root)

# Survey weighting
weight <- svydesign(id=~DhPSU, strata=~DhStrat, weights=~DhWTbrth, data=data)
options(survey.lonely.psu = "adjust")

# bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)

#####################################
##### Metric invariance testing######
######################################

# Configural model
pro_config <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + ThSDQ04 + ThSDQ09 + ThSDQ17 + ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + MhSDQ04 + MhSDQ09 + MhSDQ17 + MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ 1
  ThSDQ09 ~ 1
  ThSDQ17 ~ 1
  ThSDQ20 ~ 1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ 1
  MhSDQ09 ~ 1
  MhSDQ17 ~ 1
  MhSDQ20 ~ 1

  pro_t ~ 0*1
  pro_p ~ 1
  
  pro_t ~~ 1*pro_t
  pro_p ~~ pro_p
  pro_t ~~ pro_p
'

fit_pro_config <- cfa(pro_config, data=data, estimator="MLR")
summary(fit_pro_config, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_pro_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric Invariance
pro_metr <- '
# latent variable definitions
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ 1
  ThSDQ09 ~ 1
  ThSDQ17 ~ 1
  ThSDQ20 ~ 1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ 1
  MhSDQ09 ~ 1
  MhSDQ17 ~ 1
  MhSDQ20 ~ 1

  pro_t ~ 0*1
  pro_p ~ 1
  
  pro_t ~~ 1*pro_t
  pro_p ~~ pro_p
  pro_t ~~ pro_p
'

fit_pro_metr <- cfa(pro_metr, data=data, estimator="MLR")
summary(fit_pro_metr, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_pro_metr, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_pro_config, "cfi.scaled") - fitmeasures(fit_pro_metr, "cfi.scaled") 
fitmeasures(fit_pro_config, "rmsea.scaled") - fitmeasures(fit_pro_metr, "rmsea.scaled")
fitmeasures(fit_pro_config, "srmr") - fitmeasures(fit_pro_metr, "srmr")  

# Scalar invariance
pro_scal <- '
# latent variable definitions
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ nu3*1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ nu3*1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1

  pro_t ~ 0*1
  pro_p ~ 1
  
  pro_t ~~ 1*pro_t
  pro_p ~~ pro_p
  pro_t ~~ pro_p
'

fit_pro_scal <- cfa(pro_scal, data=data, estimator="MLR")
summary(fit_pro_scal, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_pro_scal, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_pro_metr, "cfi.scaled") - fitmeasures(fit_pro_scal, "cfi.scaled") 
fitmeasures(fit_pro_metr, "rmsea.scaled") - fitmeasures(fit_pro_scal, "rmsea.scaled")
fitmeasures(fit_pro_metr, "srmr") - fitmeasures(fit_pro_scal, "srmr") 

# See which intercept to free
lavTestScore(fit_pro_scal) # p.18 to 32 is highest chi2

## Partial scalar model
pro_scal_par <- '
# latent variable definitions
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1

  pro_t ~ 0*1
  pro_p ~ 1
  
  pro_t ~~ 1*pro_t
  pro_p ~~ pro_p
  pro_t ~~ pro_p
'

fit_pro_scal_par <- cfa(pro_scal_par, data=data, estimator="MLR")
summary(fit_pro_scal_par, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_pro_scal_par, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
print("fit differences")
fitmeasures(fit_pro_metr, "cfi.scaled") - fitmeasures(fit_pro_scal_par, "cfi.scaled") 
fitmeasures(fit_pro_metr, "rmsea.scaled") - fitmeasures(fit_pro_scal_par, "rmsea.scaled")
fitmeasures(fit_pro_metr, "srmr") - fitmeasures(fit_pro_scal_par, "srmr")

#############################################################
####################  LDS Models    #########################
#############################################################

pro_lds <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ 1
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ 1
  pro_dis ~~ pro_dis
'

## Unweighted estimates
fit_pro_lds <- sem(pro_lds, data=data, estimator="MLR")
summary(fit_pro_lds, standardized=T, fit.measures=T)

## Weighting with bootstrapped errors
boot_pro1 <- lavaan.survey(fit_pro_lds, weight_boot)
summary(boot_pro1, standardized=T, fit.measures=T)

## Compare estimates by unweighted, weighted, and bootstrapped weighted
parameterEstimates(fit_pro_lds, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]
parameterEstimates(boot_pro1, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]

## reminder: std.nox (c 13), r 33 is the Mean LDS std.
## reminder: est (c 5), r 34 is the variance.
## reminder: std.nox (c13), r 28 is the covariance std.

#################################################################################
######################### Conditional models              #######################
#################################################################################

## Conditional on the study child's biological sex, 0:male and 1: female, ref=female.

pro_lds_sex <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_female*sex
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_female*sex
  
  pro_t_male := int_pro_t
  pro_dis_male  := int_pro_dis
  pro_p_male := int_pro_t + int_pro_dis
  
  pro_t_female := int_pro_t + gamma_pro_t_female
  pro_dis_female := int_pro_dis + gamma_pro_dis_female
  pro_p_female := int_pro_t + gamma_pro_t_female + int_pro_dis + gamma_pro_dis_female
  '
# Fir summmary, unweighted
fit_pro_lds_sex <- sem(pro_lds_sex, data=data, estimator = "MLR")
summary(fit_pro_lds_sex, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_sex <- lavaan.survey(fit_pro_lds_sex, weight_boot)
summary(boot_pro_sex, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_sex)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_sex)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_sex)[c(49,51,52,54),c(4:10)]
data_plot$Gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Boy", "Girl"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

a <- ggplot(data_plot, aes(Reporter, est, colour=Gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )

## LDS conditional on parental mental health

pro_lds_depress <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_high*mcs12
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_high*mcs12
  
  pro_t_low := int_pro_t
  pro_dis_low := int_pro_dis
  pro_p_low := int_pro_t + int_pro_dis
  
  pro_t_high := int_pro_t + gamma_pro_t_high
  pro_dis_high := int_pro_dis + gamma_pro_dis_high
  pro_p_high := int_pro_t + gamma_pro_t_high + int_pro_dis + gamma_pro_dis_high
  '

#summary of SEM fit, unweighted
fit_pro_lds_depress <- sem(pro_lds_depress, data=data, estimator = "MLR")
summary(fit_pro_lds_depress, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_depress <- lavaan.survey(fit_pro_lds_depress, weight_boot)
summary(boot_pro_depress, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on maternal depression")
print("unweighted")
parameterEstimates(fit_pro_lds_depress)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_pro_depress)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_depress)[c(49,51,52,54),c(4:10)]
data_plot$MCS <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Below average", "Average"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

b <- ggplot(data_plot, aes(Reporter, est, colour=MCS)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditional on family tenure (ref = own home)

pro_lds_tenancy <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_own*tenancy
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_own*tenancy
  
  pro_t_rent := int_pro_t
  pro_dis_rent := int_pro_dis
  pro_p_rent := int_pro_t + int_pro_dis
  
  pro_t_own := int_pro_t + gamma_pro_t_own
  pro_dis_own := int_pro_dis + gamma_pro_dis_own
  pro_p_own := int_pro_t + gamma_pro_t_own + int_pro_dis + gamma_pro_dis_own
  '
#summary of SEM fit, unweighted
fit_pro_lds_tenancy <- sem(pro_lds_tenancy, data=data, estimator = "MLR")
summary(fit_pro_lds_tenancy, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_tenancy <- lavaan.survey(fit_pro_lds_tenancy, weight_boot)
summary(boot_pro_tenancy, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_tenancy)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_tenancy)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_tenancy)[c(49,51,52,54),c(4:10)]
data_plot$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

c <- ggplot(data_plot, aes(Reporter, est, colour=Tenancy)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

##################################################################
################################################################

## Conditional on parental educational attainment

pro_lds_educat <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_higher*educat
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_higher*educat
  
  pro_t_lower := int_pro_t
  pro_dis_lower := int_pro_dis
  pro_p_lower := int_pro_t + int_pro_dis
  
  pro_t_higher := int_pro_t + gamma_pro_t_higher
  pro_dis_higher := int_pro_dis + gamma_pro_dis_higher
  pro_p_higher := int_pro_t + gamma_pro_t_higher + int_pro_dis + gamma_pro_dis_higher
  '
#summary of SEM fit, unweighted
fit_pro_lds_educat <- sem(pro_lds_educat, data=data, estimator = "MLR")
summary(fit_pro_lds_educat, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_educat <- lavaan.survey(fit_pro_lds_educat, weight_boot)
summary(boot_pro_educat, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_educat)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_educat)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_educat)[c(49,51,52,54),c(4:10)]
data_plot$Education <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Below Highers", "Highers"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

d <- ggplot(data_plot, aes(Reporter, est, colour=Education)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )

## Conditional on parents reporting of subjective financial wellbeing

pro_lds_finance <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_well*finance
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_well*finance
  
  pro_t_other := int_pro_t
  pro_dis_other := int_pro_dis
  pro_p_other := int_pro_t + int_pro_dis
  
  pro_t_well := int_pro_t + gamma_pro_t_well
  pro_dis_well := int_pro_dis + gamma_pro_dis_well
  pro_p_well := int_pro_t + gamma_pro_t_well + int_pro_dis + gamma_pro_dis_well
  '
#summary of SEM fit, unweighted
fit_pro_lds_finance <- sem(pro_lds_finance, data=data, estimator = "MLR")
summary(fit_pro_lds_finance, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_finance <- lavaan.survey(fit_pro_lds_finance, weight_boot)
summary(boot_pro_finance, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on family financial health")
print("unweighted")
parameterEstimates(fit_pro_lds_finance)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_pro_finance)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_finance)[c(49,51,52,54),c(4:10)]
data_plot$Finances <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Other", "Managing well"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

e <- ggplot(data_plot, aes(Reporter, est, colour=Finances)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Condition on family composition (ref = couple household)

pro_lds_family <- '
  pro_t =~ NA*ThSDQ01 + lambda1*ThSDQ01 + lambda2*ThSDQ04 + lambda3*ThSDQ09 + lambda4*ThSDQ17 + lambda5*ThSDQ20
  pro_p =~ NA*MhSDQ01 + lambda1*MhSDQ01 + lambda2*MhSDQ04 + lambda3*MhSDQ09 + lambda4*MhSDQ17 + lambda5*MhSDQ20

  ThSDQ01 ~~ MhSDQ01
  ThSDQ04 ~~ MhSDQ04
  ThSDQ09 ~~ MhSDQ09
  ThSDQ17 ~~ MhSDQ17
  ThSDQ20 ~~ MhSDQ20
  
  ThSDQ01 ~ nu1*1
  ThSDQ04 ~ nu2*1
  ThSDQ09 ~ 1
  ThSDQ17 ~ nu4*1
  ThSDQ20 ~ nu5*1
  
  MhSDQ01 ~ nu1*1
  MhSDQ04 ~ nu2*1
  MhSDQ09 ~ 1
  MhSDQ17 ~ nu4*1
  MhSDQ20 ~ nu5*1
  
  # LDS model
  
  pro_dis =~ 1*pro_p # discrepancy of parental reports
  pro_p ~ 1*pro_t
  pro_dis ~~ pro_t
  
  pro_t ~~ pro_t
  pro_t ~ int_pro_t*1 + gamma_pro_t_couple*family
  
  pro_p ~~ 0*pro_p
  pro_p ~ 0*1
  
  pro_dis ~ int_pro_dis*1
  pro_dis ~~ pro_dis
  
  pro_dis ~ gamma_pro_dis_couple*family
  
  pro_t_single := int_pro_t
  pro_dis_single := int_pro_dis
  pro_p_single := int_pro_t + int_pro_dis
  
  pro_t_couple := int_pro_t + gamma_pro_t_couple
  pro_dis_couple := int_pro_dis + gamma_pro_dis_couple
  pro_p_couple := int_pro_t + gamma_pro_t_couple + int_pro_dis + gamma_pro_dis_couple
  '
#summary of SEM fit, unweighted
fit_pro_lds_family <- sem(pro_lds_family, data=data, estimator = "MLR")
summary(fit_pro_lds_family, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_pro_family <- lavaan.survey(fit_pro_lds_family, weight_boot)
summary(boot_pro_family, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_family)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_family)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_pro_family)[c(49,51,52,54),c(4:10)]
data_plot$Family <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Single parent", "Couple"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

f <- ggplot(data_plot, aes(Reporter, est, colour=Family)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent') 
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )



#####################################
##### Combine plots and export ######
######################################
library(cowplot)

four <- cowplot::plot_grid(a + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           b + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           c + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           d + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           e + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           f + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           ncol = 3, nrow = 2,
                           rel_heights = c(1, 1),
                           rel_widths = c(1, 1, 1)) + theme(plot.margin = margin(20, 20, 10, 10))


ggsave(file.path("C", "Users", "s1769862", "Desktop", "LDS-plots", "social-10-conditional.png"), four, width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as Rdata
saveRDS(four, file.path("C:", "Users", "s1769862", "OneDrive - University of Edinburgh", "SDQ-paper", "plotdata", "plot4.Rdata"))

