#############################################################################################
##### Parent-adolescent discrepancy modelling for the peer problems scale sweep 10       ####
#############################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw10-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw10clean(root)

# Survey weighting
options(survey.lonely.psu = "adjust")
weight <- svydesign(id=~Djpsu, strata=~Djstrata, weights=~Djwtbrthf, data=data)

## bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)

######################################
##### Metric invariance testing ######
######################################

## Configural invariance model

peer_config <- '
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + CjSDQ11 + CjSDQ14 + CjSDQ19 + CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + MjSDQ11 + MjSDQ14 + MjSDQ19 + MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ 1
CjSDQ14 ~ 1
CjSDQ19 ~ 1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ 1
MjSDQ14 ~ 1
MjSDQ19 ~ 1
MjSDQ23 ~ 1

peer_c ~ 0*1
peer_p ~ 1

peer_c ~~ 1*peer_c
peer_p ~~ peer_p
peer_c ~~ peer_p
'

print("Peer configural model CFA")
fit_peer_config <- cfa(peer_config, data=data, estimator="MLR")
#summary(fit_peer_config, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric model
## This is the equivalence of factor loadings
peer_metric <- '
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ 1
CjSDQ14 ~ 1
CjSDQ19 ~ 1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ 1
MjSDQ14 ~ 1
MjSDQ19 ~ 1
MjSDQ23 ~ 1

peer_c ~ 0*1
peer_p ~ 1

peer_c ~~ 1*peer_c
peer_p ~~ peer_p
peer_c ~~ peer_p
'
fit_peer_metric <- cfa(peer_metric, data=data, estimator="MLR")
#summary(fit_peer_metric, standardized=T, fit.measures=T)
print("Metric model CFA")
fitMeasures(fit_peer_metric, fit.measures = c("cfi.scaled","rmsea.scaled", "srmr"))

print("fit differences")
fitmeasures(fit_peer_config, "cfi.scaled") - fitmeasures(fit_peer_metric, "cfi.scaled") 
fitmeasures(fit_peer_config, "rmsea.scaled") - fitmeasures(fit_peer_metric, "rmsea.scaled")
fitmeasures(fit_peer_config, "srmr") - fitmeasures(fit_peer_metric, "srmr")


### Scalar model
## This is the equivalence of intercepts
peer_scal <-'
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ nu3*1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ nu5*1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ nu3*1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ nu5*1

peer_c ~ 0*1
peer_p ~ 1

peer_c ~~ 1*peer_c
peer_p ~~ peer_p
peer_c ~~ peer_p
'
fit_peer_scal <- cfa(peer_scal, data=data, estimator="MLR")
print("Scalar model CFA")
#summary(fit_peer_scal, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_scal, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("fit differences")

fitmeasures(fit_peer_metric, "cfi.scaled") - fitmeasures(fit_peer_scal, "cfi.scaled")
fitmeasures(fit_peer_metric, "rmsea.scaled") - fitmeasures(fit_peer_scal, "rmsea.scaled")
fitmeasures(fit_peer_metric, "srmr") - fitmeasures(fit_peer_scal, "srmr")

## Figure out which measures to free for partial scalar model
lavTestScore(fit_peer_scal)

## Trying for the partial scalar model
peer_scal_par <-'
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

peer_c ~ 0*1
peer_p ~ 1

peer_c ~~ 1*peer_c
peer_p ~~ peer_p
peer_c ~~ peer_p
'
fit_peer_scal_par <- cfa(peer_scal_par, data=data, estimator="MLR")
summary(fit_peer_scal_par, standardized=TRUE, fit.measures=TRUE)
print("Partial scalar model CFA")
fitmeasures(fit_peer_scal_par, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("fit differences")

fitmeasures(fit_peer_metric, "cfi.scaled") - fitmeasures(fit_peer_scal_par, "cfi.scaled")
fitmeasures(fit_peer_metric, "rmsea.scaled") - fitmeasures(fit_peer_scal_par, "rmsea.scaled")
fitmeasures(fit_peer_metric, "srmr") - fitmeasures(fit_peer_scal_par, "srmr")

################################################################
####################LDS Models #############################################
#########################################################################

### Latent difference score models
peer_lds <- '
  peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
  peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

  CjSDQ06 ~~ MjSDQ06
  CjSDQ11 ~~ MjSDQ11
  CjSDQ14 ~~ MjSDQ14
  CjSDQ19 ~~ MjSDQ19
  CjSDQ23 ~~ MjSDQ23

  CjSDQ06 ~ nu1*1
  CjSDQ11 ~ nu2*1
  CjSDQ14 ~ 1
  CjSDQ19 ~ nu4*1
  CjSDQ23 ~ 1

  MjSDQ06 ~ nu1*1
  MjSDQ11 ~ nu2*1
  MjSDQ14 ~ 1
  MjSDQ19 ~ nu4*1
  MjSDQ23 ~ 1

# LDS model 
  
  peer_dis =~ 1*peer_p
  peer_p ~ 1*peer_c
  peer_dis ~~ peer_c
    
  peer_c ~~ peer_c
  peer_c ~ 1
    
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
    
  peer_dis ~ 1
  peer_dis ~~ peer_dis
'

## Unweighted estimates
print("Unweighted peer problems LDS summary")
fit_peer_lds <- sem(peer_lds, data=data, estimator="MLR")
summary(fit_peer_lds, standardized=T, fit.measures=T)

## Weighting with bootstrapped errors
print("bootstrapped weighted peer problems LDS summary")
boot_peer1 <- lavaan.survey(fit_peer_lds, weight_boot)
summary(boot_peer1, standardized=T, fit.measures=T)

## Compare estimates by unweighted, weighted, and bootstrapped weighted
print("Compare LDS estimates")
print("unweighted")
parameterEstimates(fit_peer_lds, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]
print("weighted")
parameterEstimates(boot_peer1, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]

print("reminder std.nox c13 r33 is the Mean LDS std")
print("reminder est c5 r34 is the variance")
print("reminder std.nox c13 r28 is the covariance std.")

## reminder: std.nox (c 13), r 33 is the Mean LDS std.
## reminder: est (c 5), r 34 is the variance.
## reminder: std.nox (c13), r 28 is the covariance std.

#################################################################################
######################### Conditional models ########################################################
#####################################################################################


## Conditioning on YP gender

peer_lds_sex <- '

peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

# LDS model 
  
peer_dis =~ 1*peer_p # discrepancy of parent reports. this is a latent factor definition
peer_p ~ 1*peer_c
peer_dis ~~ peer_c
    
peer_c ~~ peer_c # residual correlation, these covary
peer_c ~ int_peer_c*1 + gamma_peer_c_girls*gender
    
peer_p ~~ 0*peer_p # residual correlation
peer_p ~ 0*1 # regression
    
peer_dis ~ int_peer_dis*1
peer_dis ~~ peer_dis

# conditioning on child gender

peer_dis ~ gamma_peer_dis_girls*gender

# Simple intercepts and slopes (discrepancies)

peer_c_boys := int_peer_c
peer_dis_boys := int_peer_dis
peer_p_boys := int_peer_c + int_peer_dis

peer_c_girls := int_peer_c + gamma_peer_c_girls
peer_dis_girls := int_peer_dis + gamma_peer_dis_girls
peer_p_girls := int_peer_c +gamma_peer_c_girls + int_peer_dis + gamma_peer_dis_girls
'
###### Unweighted summary statistics######
fit_peer_lds_sex <-sem(peer_lds_sex, data=data, estimator="MLR")
#print("unweighted peer conditional on gender")
#summary(fit_peer_lds_sex, standardized=TRUE, fit.measures=TRUE)

##### Weighted with bootstrapping ######
print("weighted peer conditional on gender")
boot_peer_sex <- lavaan.survey(fit_peer_lds_sex, weight_boot)
summary(boot_peer_sex, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on gender")
print("unweighted")
parameterEstimates(fit_peer_lds_sex)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_sex)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_peer_sex)[c(49,51,52,54),c(4:10)]
data_plot$Gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Boy", "Girl"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

a <- ggplot(data_plot, aes(Reporter, est, colour=Gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )


### Has the MC ever experienced several days of depression

peer_lds_distress <- ' 

peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1
    
  # LDS model 
  
    peer_dis =~ 1*peer_p # Discrepancy of parents reports
    peer_p ~ 1*peer_c
    peer_dis ~~ peer_c
    
    peer_c ~~ peer_c
    peer_c ~ int_peer_c*1 + gamma_peer_c_no*depress # conditioning the "intercept"/parent distress on reference of no depress=1
    
    peer_p ~~ 0*peer_p
    peer_p ~ 0*1
    
    peer_dis ~ int_peer_dis*1
    peer_dis ~~ peer_dis
    
  # Conditioning LDS on parent no depression
  
    peer_dis ~ gamma_peer_dis_no*depress
    
  # Simple intercepts and slopes (discrepancies)
  
    peer_c_yes := int_peer_c
    peer_dis_yes := int_peer_dis
    peer_p_yes := int_peer_c + int_peer_dis
    
    peer_c_no := int_peer_c + gamma_peer_c_no
    peer_dis_no := int_peer_dis + gamma_peer_dis_no
    peer_p_no := int_peer_c + gamma_peer_c_no + int_peer_dis + gamma_peer_dis_no '

fit_peer_lds_distress <- sem(peer_lds_distress, data=data, estimator="MLR")
#print("unweighted peer conditional on maternal depression")
#summary(fit_peer_lds_distress, standardized=T, fit.measures=T)

## Weighted with bootstrapping
boot_peer_distress <- lavaan.survey(fit_peer_lds_distress, weight_boot)
print("weighted peer conditional on maternal depression")
summary(boot_peer_distress, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on maternal depression")
print("unweighted")
parameterEstimates(fit_peer_lds_distress)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_distress)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_peer_distress)[c(49,51,52,54),c(4:10)]
data_plot$Depression <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Yes", "No"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

b <- ggplot(data_plot, aes(Reporter, est, colour=Depression)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  ) +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )


## Tenancy

peer_lds_tenancy <- ' 

peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

  # LDS model 
  
    peer_dis =~ 1*peer_p # Discrepancy of parents reports
    peer_p ~ 1*peer_c
    peer_dis ~~ peer_c
    
    peer_c ~~ peer_c
    peer_c ~ int_peer_c*1 + gamma_peer_c_own*tenancy # conditioning the "intercept"/tenancy type
    
    peer_p ~~ 0*peer_p
    peer_p ~ 0*1
    
    peer_dis ~ int_peer_dis*1
    peer_dis ~~ peer_dis
    
  # Conditioning LDS on tenancy type "Own home"
  
    peer_dis ~ gamma_peer_dis_own*tenancy
    
  # Simple intercepts and slopes (discrepancies)
  
    peer_c_rent := int_peer_c
    peer_dis_rent := int_peer_dis
    peer_p_rent := int_peer_c + int_peer_dis
    
    peer_c_own := int_peer_c + gamma_peer_c_own
    peer_dis_own := int_peer_dis + gamma_peer_dis_own
    peer_p_own := int_peer_c + gamma_peer_c_own + int_peer_dis + gamma_peer_dis_own '

fit_peer_lds_tenancy <- sem(peer_lds_tenancy, data=data, estimator="MLR")
#print("unweighted peer conditional on household tenancy")
#summary(fit_peer_lds_tenancy, standardized=T, fit.measures=T)

###### Weighted with bootstrapping######
boot_peer_tenancy <- lavaan.survey(fit_peer_lds_tenancy, weight_boot)
print("weighted peer conditional on household tenancy")
summary(boot_peer_tenancy, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household tenancy")
print("unweighted")
parameterEstimates(fit_peer_lds_tenancy)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_tenancy)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_peer_tenancy)[c(49,51,52,54),c(4:10)]
data_plot$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own home"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

c <- ggplot(data_plot, aes(Reporter, est, colour=Tenancy)) +
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) + theme_hc()+ scale_colour_hc()+
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  ) +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )


## Conditioning on family composition

peer_lds_family <- '
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

# LDS model 
  
peer_dis =~ 1*peer_p # discrepancy of parent reports. this is a latent factor definition
peer_p ~ 1*peer_c
peer_dis ~~ peer_c
    
peer_c ~~ peer_c
peer_c ~ int_peer_c*1 + gamma_peer_c_couple*family
    
peer_p ~~ 0*peer_p # residual correlation
peer_p ~ 0*1 # regression
    
peer_dis ~ int_peer_dis*1
peer_dis ~~ peer_dis

# conditioning on household composition

peer_dis ~ gamma_peer_dis_couple*family

# Simple intercepts and slopes (discrepancies)

peer_c_single := int_peer_c
peer_dis_single := int_peer_dis
peer_p_single := int_peer_c + int_peer_dis

peer_c_couple := int_peer_c + gamma_peer_c_couple
peer_dis_couple := int_peer_dis + gamma_peer_dis_couple
peer_p_couple := int_peer_c +gamma_peer_c_couple + int_peer_dis + gamma_peer_dis_couple
'
## Unweighted summary statistics
fit_peer_lds_family <-sem(peer_lds_family, data=data, estimator="MLR")
print("unweighted peer conditional on family composition")
summary(fit_peer_lds_family, standardized=TRUE, fit.measures=TRUE)

####### Weighted with bootstrapping####
boot_peer_family <- lavaan.survey(fit_peer_lds_family, weight_boot)
print("weighted peer conditional on household composition")
summary(boot_peer_family, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on composition of the family")
print("unweighted")
parameterEstimates(fit_peer_lds_family)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_family)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_peer_family)[c(49,51,52,54),c(4:10)]
data_plot$Family <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Single parent", "Couple"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

d <- ggplot(data_plot, aes(Reporter, est, colour=Family)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )


## Conditioning on language of the home
peer_lds_lang <- '
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

# LDS model 
  
peer_dis =~ 1*peer_p # discrepancy of parent reports. this is a latent factor definition
peer_p ~ 1*peer_c
peer_dis ~~ peer_c
    
peer_c ~~ peer_c # residual correlation, these covary
peer_c ~ int_peer_c*1 + gamma_peer_c_english*lang
    
peer_p ~~ 0*peer_p # residual correlation
peer_p ~ 0*1 # regression
    
peer_dis ~ int_peer_dis*1
peer_dis ~~ peer_dis

# conditioning on household language

peer_dis ~ gamma_peer_dis_english*lang

# Simple intercepts and slopes (discrepancies)

peer_c_other := int_peer_c
peer_dis_other := int_peer_dis
peer_p_other := int_peer_c + int_peer_dis

peer_c_english := int_peer_c + gamma_peer_c_english
peer_dis_english := int_peer_dis + gamma_peer_dis_english
peer_p_english := int_peer_c +gamma_peer_c_english + int_peer_dis + gamma_peer_dis_english
'
## Unweighted summary statistics
fit_peer_lds_lang <-sem(peer_lds_lang, data=data, estimator="MLR")
print("unweighted peer conditional on household language")
summary(fit_peer_lds_lang, standardized=TRUE, fit.measures=TRUE)

####### Weighted with bootstrapping####
boot_peer_lang <- lavaan.survey(fit_peer_lds_lang, weight_boot)
print("weighted peer conditional on household language")
summary(boot_peer_lang, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household language")
print("unweighted")
parameterEstimates(fit_peer_lds_lang)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_lang)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_e <- parameterEstimates(boot_peer_lang)[c(49,51,52,54),c(4:10)]
data_plot_e$Language <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Only English"))
data_plot_e$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_e

e <- ggplot(data_plot_e, aes(Reporter, est, colour=Language)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditioning on subjective household finances

peer_lds_finance <- '
peer_c =~ NA*CjSDQ06 + lambda1*CjSDQ06 + lambda2*CjSDQ11 + lambda3*CjSDQ14 + lambda4*CjSDQ19 + lambda5*CjSDQ23
peer_p =~ NA*MjSDQ06 + lambda1*MjSDQ06 + lambda2*MjSDQ11 + lambda3*MjSDQ14 + lambda4*MjSDQ19 + lambda5*MjSDQ23

CjSDQ06 ~~ MjSDQ06
CjSDQ11 ~~ MjSDQ11
CjSDQ14 ~~ MjSDQ14
CjSDQ19 ~~ MjSDQ19
CjSDQ23 ~~ MjSDQ23

CjSDQ06 ~ nu1*1
CjSDQ11 ~ nu2*1
CjSDQ14 ~ 1
CjSDQ19 ~ nu4*1
CjSDQ23 ~ 1

MjSDQ06 ~ nu1*1
MjSDQ11 ~ nu2*1
MjSDQ14 ~ 1
MjSDQ19 ~ nu4*1
MjSDQ23 ~ 1

# LDS model 
  
peer_dis =~ 1*peer_p # discrepancy of parent reports. this is a latent factor definition
peer_p ~ 1*peer_c
peer_dis ~~ peer_c
    
peer_c ~~ peer_c
peer_c ~ int_peer_c*1 + gamma_peer_c_well*finance
    
peer_p ~~ 0*peer_p # residual correlation
peer_p ~ 0*1 # regression
    
peer_dis ~ int_peer_dis*1
peer_dis ~~ peer_dis

# conditioning on household finances

peer_dis ~ gamma_peer_dis_well*finance

# Simple intercepts and slopes (discrepancies)

peer_c_other := int_peer_c
peer_dis_other := int_peer_dis
peer_p_other := int_peer_c + int_peer_dis

peer_c_well := int_peer_c + gamma_peer_c_well
peer_dis_well := int_peer_dis + gamma_peer_dis_well
peer_p_well := int_peer_c +gamma_peer_c_well + int_peer_dis + gamma_peer_dis_well
'
## Unweighted summary statistics
fit_peer_lds_finance <-sem(peer_lds_finance, data=data, estimator="MLR")
print("unweighted peer conditional on household finances")
summary(fit_peer_lds_finance, standardized=TRUE, fit.measures=TRUE)

##### Weighted with bootstrapping ####
boot_peer_finance <- lavaan.survey(fit_peer_lds_finance, weight_boot)
print("weighted peer conditional on household finances")
summary(boot_peer_finance, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household financial status")
print("unweighted")
parameterEstimates(fit_peer_lds_finance)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_peer_finance)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_f <- parameterEstimates(boot_peer_finance)[c(49,51,52,54),c(4:10)]
data_plot_f$Finances <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Managing well"))
data_plot_f$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_f

f <- ggplot(data_plot_f, aes(Reporter, est, colour=Finances)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Peer factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )


#####################################
##### Combine plots and export ######
######################################

library(cowplot)

three <- cowplot::plot_grid(a + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           b + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           c + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           d + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           e + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           f + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                           ncol = 3, nrow = 2,
                           rel_heights = c(1, 1),
                           rel_widths = c(1, 1, 1)) + theme(plot.margin = margin(20, 20, 10, 10))

## save plot as png
ggsave(file.path("C", "Users", "s1769862", "Desktop", "LDS-plots", "peer-14-conditional.png"), three,  width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as Rdata
saveRDS(three, file.path("C:", "Users", "s1769862", "OneDrive - University of Edinburgh", "SDQ-paper", "plotdata", "plot3.Rdata"))

