################################################################################################
##### Parent-adolescent discrepancy modelling for the emotional difficulties scale sweep 10 ####
################################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw10-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw10clean(root)


############################################
###########   Sample statistics   ##########
############################################

## gender
summary(data$gender, na.rm=T)

## tenancy
summary(data$tenancy, na.rm=T)

## family composition
summary(data$family, na.rm=T)

## finances
summary(data$finance, na.rm=T)

## language
summary(data$lang, na.rm=T)
table(data$lang)

## maternal depression
mean(data$depress, na.rm=T)
summary(data$depress, na.rm=T)


# Survey weighting
options(survey.lonely.psu = "adjust")
weight <- svydesign(id=~Djpsu, strata=~Djstrata, weights=~Djwtbrthf, data=data)

## bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)

#####################################
##### Metric invariance testing ######
######################################

## Configural invariance model
emo_config <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + CjSDQ08 + CjSDQ13 + CjSDQ16 + CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + MjSDQ08 + MjSDQ13 + MjSDQ16 + MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ 1
CjSDQ13 ~ 1
CjSDQ16 ~ 1
CjSDQ24 ~ 1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ 1
MjSDQ13 ~ 1
MjSDQ16 ~ 1
MjSDQ24 ~ 1

emo_c ~ 0*1
emo_p ~ 1

emo_c ~~ 1*emo_c
emo_p ~~ emo_p
emo_c ~~ emo_p
'

print("Emotional configural model CFA")
fit_emo_config <- cfa(emo_config, data=data, estimator = "MLR")
#summary(fit_emo_config, standardized=TRUE, fit.measures=TRUE)
fitMeasures(fit_emo_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric model
emo_metr <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ 1
CjSDQ13 ~ 1
CjSDQ16 ~ 1
CjSDQ24 ~ 1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ 1
MjSDQ13 ~ 1
MjSDQ16 ~ 1
MjSDQ24 ~ 1

emo_c ~ 0*1
emo_p ~ 1

emo_c ~~ 1*emo_c
emo_p ~~ emo_p
emo_c ~~ emo_p
'
fit_emo_metr <- cfa(emo_metr, data=data, estimator = "MLR")
#summary(fit_emo_metr, standardized=TRUE, fit.measures=TRUE)
print("Metric model CFA")
fitMeasures(fit_emo_metr, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("fit differences")
fitmeasures(fit_emo_config, "cfi.scaled") - fitmeasures(fit_emo_metr, "cfi.scaled") 
fitmeasures(fit_emo_config, "rmsea.scaled") - fitmeasures(fit_emo_metr, "rmsea.scaled")
fitmeasures(fit_emo_config, "srmr") - fitmeasures(fit_emo_metr, "srmr")

## Scalar model
emo_scal <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ nu4*1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ nu4*1
MjSDQ24 ~ nu5*1

emo_c ~ 0*1
emo_p ~ 1

emo_c ~~ 1*emo_c
emo_p ~~ emo_p
emo_c ~~ emo_p
'
fit_emo_scal <- cfa(emo_scal, data=data, estimator = "MLR")
print("Scalar model CFA")
#summary(fit_emo_scal, standardized=TRUE, fit.measures=TRUE)
fitMeasures(fit_emo_scal, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("fit differences")

fitmeasures(fit_emo_metr, "cfi.scaled") - fitmeasures(fit_emo_scal, "cfi.scaled") 
fitmeasures(fit_emo_metr, "rmsea.scaled") - fitmeasures(fit_emo_scal, "rmsea.scaled")
fitmeasures(fit_emo_metr, "srmr") - fitmeasures(fit_emo_scal, "srmr")

## Looking for a parameter to free
lavTestScore(fit_emo_scal)
## free p.19 to 24

## Partial scalar model
emo_scal_par <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1

emo_c ~ 0*1
emo_p ~ 1

emo_c ~~ 1*emo_c
emo_p ~~ emo_p
emo_c ~~ emo_p
'
fit_emo_scal_par <- cfa(emo_scal_par, data=data, estimator = "MLR")
#summary(fit_emo_scal_par, standardized=TRUE, fit.measures=TRUE)
print("Partial scalar model CFA")
fitMeasures(fit_emo_scal_par, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("fit differences")

fitmeasures(fit_emo_metr, "cfi.scaled") - fitmeasures(fit_emo_scal_par, "cfi.scaled") 
fitmeasures(fit_emo_metr, "rmsea.scaled") - fitmeasures(fit_emo_scal_par, "rmsea.scaled")
fitmeasures(fit_emo_metr, "srmr") - fitmeasures(fit_emo_scal_par, "srmr")

#########################################################################################
############################### LDS models ################################################
##########################################################################################


emo_lds <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1

# LDS model 
  
emo_dis =~ 1*emo_p # discrepancy of parent reoprts latent factor definition
emo_p ~ 1*emo_c # regression
emo_dis ~~ emo_c # residual correlation
    
emo_c ~~ emo_c # residual correlation
emo_c ~ 1 # regression
    
emo_p ~~ 0*emo_p # residual correlation
emo_p ~ 0*1 # regression
    
emo_dis ~ 1 # regression
emo_dis ~~ emo_dis
'
### Unweighed summary
print("Unweighted emotion symptoms LDS summary")
fit_emo_lds <- sem(emo_lds, data=data, estimator="MLR")
summary(fit_emo_lds, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapped errors
print("bootstrapped weighted emotion symptoms LDS summary")
boot_emo1 <- lavaan.survey(fit_emo_lds, weight_boot)
summary(boot_emo1, standardized=T, fit.measures=T)

## Compare estimates by unweighted, weighted, and bootstrapped weighted
print("Compare LDS estimates")
print("unweighted")
parameterEstimates(fit_emo_lds, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]
print("weighted")
parameterEstimates(boot_emo1, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]

## Fit measures
fitMeasures(fit_emo_lds, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))
fitMeasures(boot_emo1, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

print("reminder std.nox c13 r33 is the Mean LDS std")
print("reminder est c5 r34 is the variance")
print("reminder std.nox c13 r28 is the covariance std.")

#################################################################################
######################### Conditional models ########################################################
#####################################################################################


### Conditional models

### Adolescent gender
##Remember that CjGenId 0 = boy, 1 = girl

emo_lds_sex <- ' 

  # Measurement model, partial scalar

emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1
    
# LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_girls*gender # conditioning the "intercept"/childs factor on their gender
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis
    
# Conditioning LDS on childs gender
  
emo_dis ~ gamma_emo_dis_girls*gender
    
 # Simple intercepts and slopes (discrepancies)
  
emo_c_boys := int_emo_c # is defined as
emo_dis_boys := int_emo_dis
emo_p_boys := int_emo_c + int_emo_dis
    
emo_c_girls := int_emo_c + gamma_emo_c_girls
emo_dis_girls := int_emo_dis + gamma_emo_dis_girls
emo_p_girls := int_emo_c + gamma_emo_c_girls + int_emo_dis + gamma_emo_dis_girls '

#Unweighted estimates
fit_emo_lds_sex <- sem(emo_lds_sex, data=data, estimator="MLR")
#print("unweighted emotion conditional on gender")
#summary(fit_emo_lds_sex, standardized=T, fit.measures=T)

###### Weighted with bootstrapping #####
print("weighted emotion conditional on gender")
boot_emo_sex <- lavaan.survey(fit_emo_lds_sex, weight_boot)
summary(boot_emo_sex, standardized=T, fit.measures=T)

###### Comparing parameter estimates ####
print("Compare estimates conditional on gender")
print("unweighted")
parameterEstimates(fit_emo_lds_sex)[c(49,51,52,54),c(4:10)]
#parameterEstimates(fit_emo_lds_sex_svy)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_sex)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_sex)[c(49,51,52,54),c(4:10)]
data_plot$Gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Boy", "Girl"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

a <- ggplot(data_plot, aes(Reporter, est, colour=Gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )


##### Has the MC ever experienced several days of depression

emo_lds_distress <- '
emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1

  # LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_no*depress # conditioning the "intercept"/parent distress on reference of no depress=1
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis
    
  # Conditioning LDS on parent no depression
  
emo_dis ~ gamma_emo_dis_no*depress
    
  # Simple intercepts and slopes (discrepancies)
  
emo_c_yes := int_emo_c
emo_dis_yes := int_emo_dis
emo_p_yes := int_emo_c + int_emo_dis
    
emo_c_no := int_emo_c + gamma_emo_c_no
emo_dis_no := int_emo_dis + gamma_emo_dis_no
emo_p_no := int_emo_c + gamma_emo_c_no + int_emo_dis + gamma_emo_dis_no
'

fit_emo_lds_distress <- sem(emo_lds_distress, data=data, estimator="MLR")
#print("unweighted emotion conditional on maternal depression")
summary(fit_emo_lds_distress, standardized=T, fit.measures=T)

##### Weighted with bootstrapping#######
boot_emo_distress <- lavaan.survey(fit_emo_lds_distress, weight_boot)
print("weighted emotion conditional on maternal depression")
summary(boot_emo_distress, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on maternal depression")
print("unweighted")
parameterEstimates(fit_emo_lds_distress)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_distress)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_distress)[c(49,51,52,54),c(4:10)]
data_plot$Depression <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Yes", "No"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

b <- ggplot(data_plot, aes(Reporter, est, colour=Depression)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), linewidth=.3, width=.2) +
  xlab("") + 
  ylab("Emotional factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Tenancy

emo_lds_tenancy <- ' 
   # Measurement model, partial scalar

emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1
    
  # LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_own*tenancy # conditioning the "intercept"/tenancy type
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1 # specify mean of latent true score
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis
    
  # Conditioning LDS on tenancy type "Own home"
  
emo_dis ~ gamma_emo_dis_own*tenancy
    
  # Simple intercepts and slopes (discrepancies)
  
emo_c_rent := int_emo_c
emo_dis_rent := int_emo_dis
emo_p_rent := int_emo_c + int_emo_dis
    
emo_c_own := int_emo_c + gamma_emo_c_own
emo_dis_own := int_emo_dis + gamma_emo_dis_own
emo_p_own := int_emo_c + gamma_emo_c_own + int_emo_dis + gamma_emo_dis_own
'

fit_emo_lds_tenancy <- sem(emo_lds_tenancy, data=data, estimator="MLR")
#print("unweighted emotion conditional on household tenancy")
#summary(fit_emo_lds_tenancy, standardized=T, fit.measures=T)

###### Weighted with bootstrapping #####
boot_emo_tenancy <- lavaan.survey(fit_emo_lds_tenancy, weight_boot)
print("weighted emotion conditional on household tenancy")
summary(boot_emo_tenancy, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household tenancy")
print("unweighted")
parameterEstimates(fit_emo_lds_tenancy)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_tenancy)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_tenancy)[c(49,51,52,54),c(4:10)]
data_plot$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own home"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

c <- ggplot(data_plot, aes(Reporter, est, colour=Tenancy)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), linewidth=.3, width=.2) +
  xlab("") + 
  ylab("Emotional factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditional on household composition

emo_lds_family <- ' 

  # Measurement model, partial scalar

emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1
    
    # LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_couple*family
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis

# conditioning on household composition
  
emo_dis ~ gamma_emo_dis_couple*family
    
  # Simple intercepts and slopes (discrepancies)
  
emo_c_single := int_emo_c
emo_dis_single := int_emo_dis
emo_p_single := int_emo_c + int_emo_dis
    
emo_c_couple := int_emo_c + gamma_emo_c_couple
emo_dis_couple := int_emo_dis + gamma_emo_dis_couple
emo_p_couple := int_emo_c + gamma_emo_c_couple + int_emo_dis + gamma_emo_dis_couple
'

fit_emo_lds_family <- sem(emo_lds_family, data=data, estimator="MLR")
print("unweighted emotion conditional on household composition")
summary(fit_emo_lds_family, standardized=T, fit.measures=T)

###### Weighted with bootstrapping ####
boot_emo_family <- lavaan.survey(fit_emo_lds_family, weight_boot)
print("weighted emotionconditional on household composition")
summary(boot_emo_family, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household composition")
print("unweighted")
parameterEstimates(fit_emo_lds_family)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_family)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_family)[c(49,51,52,54),c(4:10)]
data_plot$Family <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Single parent", "Couple"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot

d <- ggplot(data_plot, aes(Reporter, est, colour=Family)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )

## Conditional on household language
emo_lds_lang <- ' 

  # Measurement model, partial scalar

emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1
    
    # LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_english*lang
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis

# conditioning on household language
  
emo_dis ~ gamma_emo_dis_english*lang
    
  # Simple intercepts and slopes (discrepancies)
  
emo_c_other := int_emo_c
emo_dis_other := int_emo_dis
emo_p_other := int_emo_c + int_emo_dis
    
emo_c_english := int_emo_c + gamma_emo_c_english
emo_dis_english := int_emo_dis + gamma_emo_dis_english
emo_p_english := int_emo_c + gamma_emo_c_english + int_emo_dis + gamma_emo_dis_english
'

fit_emo_lds_lang <- sem(emo_lds_lang, data=data, estimator="MLR")
print("unweighted emotion conditional on household language")
summary(fit_emo_lds_lang, standardized=T, fit.measures=T)

###### Weighted with bootstrapping ####
boot_emo_lang <- lavaan.survey(fit_emo_lds_lang, weight_boot)
print("weighted emotionconditional on household language")
summary(boot_emo_lang, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household language")
print("unweighted")
parameterEstimates(fit_emo_lds_lang)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_lang)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_e <- parameterEstimates(boot_emo_lang)[c(49,51,52,54),c(4:10)]
data_plot_e$Language <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Only English"))
data_plot_e$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_e

e <- ggplot(data_plot_e, aes(Reporter, est, colour=Language)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditional on subjective household finances

emo_lds_finance <- ' 

  # Measurement model, partial scalar

emo_c =~ NA*CjSDQ03 + lambda1*CjSDQ03 + lambda2*CjSDQ08 + lambda3*CjSDQ13 + lambda4*CjSDQ16 + lambda5*CjSDQ24
emo_p =~ NA*MjSDQ03 + lambda1*MjSDQ03 + lambda2*MjSDQ08 + lambda3*MjSDQ13 + lambda4*MjSDQ16 + lambda5*MjSDQ24

CjSDQ03 ~~ MjSDQ03
CjSDQ08 ~~ MjSDQ08
CjSDQ13 ~~ MjSDQ13
CjSDQ16 ~~ MjSDQ16
CjSDQ24 ~~ MjSDQ24

CjSDQ03 ~ nu1*1
CjSDQ08 ~ nu2*1
CjSDQ13 ~ nu3*1
CjSDQ16 ~ 1
CjSDQ24 ~ nu5*1

MjSDQ03 ~ nu1*1
MjSDQ08 ~ nu2*1
MjSDQ13 ~ nu3*1
MjSDQ16 ~ 1
MjSDQ24 ~ nu5*1
    
    # LDS model 
  
emo_dis =~ 1*emo_p # Discrepancy of parents reports
emo_p ~ 1*emo_c
emo_dis ~~ emo_c
    
emo_c ~~ emo_c
emo_c ~ int_emo_c*1 + gamma_emo_c_well*finance
    
emo_p ~~ 0*emo_p
emo_p ~ 0*1
    
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis

# conditioning on household finances
  
emo_dis ~ gamma_emo_dis_well*finance
    
  # Simple intercepts and slopes (discrepancies)
  
emo_c_other := int_emo_c
emo_dis_other := int_emo_dis
emo_p_other := int_emo_c + int_emo_dis
    
emo_c_well := int_emo_c + gamma_emo_c_well
emo_dis_well := int_emo_dis + gamma_emo_dis_well
emo_p_well := int_emo_c + gamma_emo_c_well + int_emo_dis + gamma_emo_dis_well '

fit_emo_lds_finance <- sem(emo_lds_finance, data=data, estimator="MLR")
print("unweighted emotion conditional on household finances")
summary(fit_emo_lds_finance, standardized=T, fit.measures=T)

##### Weighted with bootstrapping #####
boot_emo_finance <- lavaan.survey(fit_emo_lds_finance, weight_boot)
print("weighted emotion conditional on household finances")
summary(boot_emo_finance, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on household financial status")
print("unweighted")
parameterEstimates(fit_emo_lds_finance)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_finance)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_f <- parameterEstimates(boot_emo_finance)[c(49,51,52,54),c(4:10)]
data_plot_f$Finances <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Managing well"))
data_plot_f$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_f

f <- ggplot(data_plot_f, aes(Reporter, est, colour=Finances)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )


#####################################
##### Combine plots and export ######
######################################

library(cowplot)

two <- cowplot::plot_grid(a + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          b + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          c + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          d + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          e + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          f + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          ncol = 3, nrow = 2,
                          rel_heights = c(1, 1),
                          rel_widths = c(1, 1, 1)) + theme(plot.margin = margin(20, 20, 10, 10))



ggsave("C:/Users/s1769862/Desktop/LDs-plots/emo-14-conditional.png", plot=two, width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as png
ggsave(file.path("C", "Users", "s1769862", "Desktop", "LDS-plots", "emo-14-conditional.png"), two,  width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as Rdata
saveRDS(two, file.path("C:", "Users", "s1769862", "OneDrive - University of Edinburgh", "SDQ-paper", "plotdata", "plot2.Rdata"))
