################################################################################################
##### Parent-teacher discrepancy modelling for the peer problemssweep 8                    ####
################################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw8-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw8clean(root)

# Survey weighting
weight <- svydesign(id=~DhPSU, strata=~DhStrat, weights=~DhWTbrth, data=data)
options(survey.lonely.psu = "adjust")

# bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)


#####################################
##### Metric invariance testing######
######################################


# Configural model

peer_config <- '
# latent variable definitions (i think)
  peer_t =~ NA*ThSDQ01 + lambda1*ThSDQ06 + ThSDQ11 + ThSDQ14 + ThSDQ19 + ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + MhSDQ11 + MhSDQ14 + MhSDQ19 + MhSDQ23

# variances and covariances (i think)
  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ 1
  ThSDQ14 ~ 1
  ThSDQ19 ~ 1
  ThSDQ23 ~ 1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ 1
  MhSDQ14 ~ 1
  MhSDQ19 ~ 1
  MhSDQ23 ~ 1

  peer_t ~ 0*1
  peer_p ~ 1
  
  peer_t ~~ 1*peer_t
  peer_p ~~ peer_p
  peer_t ~~ peer_p
'

fit_peer_config <- cfa(peer_config, data=data, estimator="MLR")
summary(fit_peer_config, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric Invariance
peer_metr <- '
# latent variable definitions
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ 1
  ThSDQ14 ~ 1
  ThSDQ19 ~ 1
  ThSDQ23 ~ 1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ 1
  MhSDQ14 ~ 1
  MhSDQ19 ~ 1
  MhSDQ23 ~ 1

  peer_t ~ 0*1
  peer_p ~ 1
  
  peer_t ~~ 1*peer_t
  peer_p ~~ peer_p
  peer_t ~~ peer_p
'

fit_peer_metr <- cfa(peer_metr, data=data, estimator="MLR")
summary(fit_peer_metr, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_metr, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_peer_config, "cfi.scaled") - fitmeasures(fit_peer_metr, "cfi.scaled") 
fitmeasures(fit_peer_config, "rmsea.scaled") - fitmeasures(fit_peer_metr, "rmsea.scaled")
fitmeasures(fit_peer_config, "srmr") - fitmeasures(fit_peer_metr, "srmr")  

# Scalar invariance
peer_scal <- '
# latent variable definitions
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ nu3*1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ nu3*1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1

  peer_t ~ 0*1
  peer_p ~ 1
  
  peer_t ~~ 1*peer_t
  peer_p ~~ peer_p
  peer_t ~~ peer_p
'
## Model fit
fit_peer_scal <- cfa(peer_scal, data=data, estimator="MLR")
summary(fit_peer_scal, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_scal, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_peer_metr, "cfi.scaled") - fitmeasures(fit_peer_scal, "cfi.scaled") 
fitmeasures(fit_peer_metr, "rmsea.scaled") - fitmeasures(fit_peer_scal, "rmsea.scaled")
fitmeasures(fit_peer_metr, "srmr") - fitmeasures(fit_peer_scal, "srmr") 

# See which intercept to free
lavTestScore(fit_peer_scal) # p.18 to 32 is highest

## Partial scalar model
peer_scal_par <- '
# latent variable definitions
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1

  peer_t ~ 0*1
  peer_p ~ 1
  
  peer_t ~~ 1*peer_t
  peer_p ~~ peer_p
  peer_t ~~ peer_p
'
## Model fit
fit_peer_scal_par <- cfa(peer_scal_par, data=data, estimator="MLR")
summary(fit_peer_scal_par, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_peer_scal_par, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_peer_metr, "cfi.scaled") - fitmeasures(fit_peer_scal_par, "cfi.scaled") 
fitmeasures(fit_peer_metr, "rmsea.scaled") - fitmeasures(fit_peer_scal_par, "rmsea.scaled")
fitmeasures(fit_peer_metr, "srmr") - fitmeasures(fit_peer_scal_par, "srmr")

# Latent difference score

peer_lds <- '
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1
  
  # LDS model
  
  peer_dis =~ 1*peer_p # discrepancy of parental reports
  peer_p ~ 1*peer_t
  peer_dis ~~ peer_t
  
  peer_t ~~ peer_t
  peer_t ~ 1
  
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
  
  peer_dis ~ 1
  peer_dis ~~ peer_dis
'

## Unweighted estimates
fit_peer_lds <- sem(peer_lds, data=data, estimator="MLR")
summary(fit_peer_lds, standardized=T, fit.measures=T)

##Weighted statistics
fit_peer_lds_svy <- lavaan.survey(fit_peer_lds, weight, estimator="ML")
summary(fit_peer_lds_svy, standardized=T, fit.measures=T)

## Conditional models
## Adolescent sex. sex, 0:male and 1: female, ref=female.

peer_lds_sex <- '
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1
  
  # LDS model
  
  peer_dis =~ 1*peer_p # discrepancy of parental reports
  peer_p ~ 1*peer_t
  peer_dis ~~ peer_t
  
  peer_t ~~ peer_t
  peer_t ~ int_peer_t*1 + gamma_peer_t_female*sex
  
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
  
  peer_dis ~ int_peer_dis*1
  peer_dis ~~ peer_dis
  
  peer_dis ~ gamma_peer_dis_female*sex
  
  peer_t_male := int_peer_t
  peer_dis_male  := int_peer_dis
  peer_p_male := int_peer_t + int_peer_dis
  
  peer_t_female := int_peer_t + gamma_peer_t_female
  peer_dis_female := int_peer_dis + gamma_peer_dis_female
  peer_p_female := int_peer_t + gamma_peer_t_female + int_peer_dis + gamma_peer_dis_female
  '
#summary of SEM fit, unweighted
fit_peer_lds_sex <- sem(peer_lds_sex, data=data, estimator = "MLR")
summary(fit_peer_lds_sex, standardized=TRUE, fit.measures=TRUE)

## weighted statistics
fit_peer_lds_sex_svy <- lavaan.survey(fit_peer_lds_sex, weight, estimator="ML")
summary(fit_peer_lds_sex_svy, standardized=TRUE, fit.measures=TRUE)

parameterEstimates(fit_peer_lds_sex_svy)

#Plot weighted
data_plot <- parameterEstimates(fit_peer_lds_sex_svy)[c(49,51,52,54),c(4:10)]
data_plot$gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Male", "Female"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "MC"))
data_plot

ggplot(data_plot, aes(Reporter, est, colour=gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.5) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4)

## Maternal psychological distress

peer_lds_depress <- '
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1
  
  # LDS model
  
  peer_dis =~ 1*peer_p # discrepancy of parental reports
  peer_p ~ 1*peer_t
  peer_dis ~~ peer_t
  
  peer_t ~~ peer_t
  peer_t ~ int_peer_t*1 + gamma_peer_t_high*mcs12
  
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
  
  peer_dis ~ int_peer_dis*1
  peer_dis ~~ peer_dis
  
  peer_dis ~ gamma_peer_dis_high*mcs12
  
  peer_t_low := int_peer_t
  peer_dis_low := int_peer_dis
  peer_p_low := int_peer_t + int_peer_dis
  
  peer_t_high := int_peer_t + gamma_peer_t_high
  peer_dis_high := int_peer_dis + gamma_peer_dis_high
  peer_p_high := int_peer_t + gamma_peer_t_high + int_peer_dis + gamma_peer_dis_high
  '
#summary of SEM fit, unweighted
fit_peer_lds_depress <- sem(peer_lds_depress, data=data, estimator = "MLR")
summary(fit_peer_lds_depress, standardized=TRUE, fit.measures=TRUE)

## weighted statistics
fit_peer_lds_depress_svy <- lavaan.survey(fit_peer_lds_depress, weight, estimator="ML")
summary(fit_peer_lds_depress_svy, standardized=TRUE, fit.measures=TRUE)

parameterEstimates(fit_peer_lds_depress_svy)

#Plot weighted
data_plot <- parameterEstimates(fit_peer_lds_depress_svy)[c(49,51,52,54),c(4:10)]
data_plot$MCS <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Below average", "Average"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "MC"))
data_plot

ggplot(data_plot, aes(Reporter, est, colour=MCS)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.5) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4)

## Conditional on tenancy status

peer_lds_tenan <- '
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1
  
  # LDS model
  
  peer_dis =~ 1*peer_p # discrepancy of parental reports
  peer_p ~ 1*peer_t
  peer_dis ~~ peer_t
  
  peer_t ~~ peer_t
  peer_t ~ int_peer_t*1 + gamma_peer_t_own*tenancy
  
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
  
  peer_dis ~ int_peer_dis*1
  peer_dis ~~ peer_dis
  
  peer_dis ~ gamma_peer_dis_own*tenancy
  
  peer_t_rent := int_peer_t
  peer_dis_rent := int_peer_dis
  peer_p_rent := int_peer_t + int_peer_dis
  
  peer_t_own := int_peer_t + gamma_peer_t_own
  peer_dis_own := int_peer_dis + gamma_peer_dis_own
  peer_p_own := int_peer_t + gamma_peer_t_own + int_peer_dis + gamma_peer_dis_own
  '
#summary of SEM fit, unweighted
fit_peer_lds_tenan <- sem(peer_lds_tenan, data=data, estimator = "MLR")
summary(fit_peer_lds_tenan, standardized=TRUE, fit.measures=TRUE)

## weighted statistics
fit_peer_lds_tenan_svy <- lavaan.survey(fit_peer_lds_tenan, weight, estimator="ML")
summary(fit_peer_lds_tenan_svy, standardized=TRUE, fit.measures=TRUE)

parameterEstimates(fit_peer_lds_tenan_svy)

#Plot weighted
data_plot <- parameterEstimates(fit_peer_lds_tenan_svy)[c(49,51,52,54),c(4:10)]
data_plot$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "MC"))
data_plot

ggplot(data_plot, aes(Reporter, est, colour=Tenancy)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.5) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4)

## Conditional on MC education level
peer_lds_educat <- '
  peer_t =~ NA*ThSDQ06 + lambda1*ThSDQ06 + lambda2*ThSDQ11 + lambda3*ThSDQ14 + lambda4*ThSDQ19 + lambda5*ThSDQ23
  peer_p =~ NA*MhSDQ06 + lambda1*MhSDQ06 + lambda2*MhSDQ11 + lambda3*MhSDQ14 + lambda4*MhSDQ19 + lambda5*MhSDQ23

  ThSDQ06 ~~ MhSDQ06
  ThSDQ11 ~~ MhSDQ11
  ThSDQ14 ~~ MhSDQ14
  ThSDQ19 ~~ MhSDQ19
  ThSDQ23 ~~ MhSDQ23
  
  ThSDQ06 ~ nu1*1
  ThSDQ11 ~ nu2*1
  ThSDQ14 ~ 1
  ThSDQ19 ~ nu4*1
  ThSDQ23 ~ nu5*1
  
  MhSDQ06 ~ nu1*1
  MhSDQ11 ~ nu2*1
  MhSDQ14 ~ 1
  MhSDQ19 ~ nu4*1
  MhSDQ23 ~ nu5*1
  
  # LDS model
  
  peer_dis =~ 1*peer_p # discrepancy of parental reports
  peer_p ~ 1*peer_t
  peer_dis ~~ peer_t
  
  peer_t ~~ peer_t
  peer_t ~ int_peer_t*1 + gamma_peer_t_higher*educat
  
  peer_p ~~ 0*peer_p
  peer_p ~ 0*1
  
  peer_dis ~ int_peer_dis*1
  peer_dis ~~ peer_dis
  
  peer_dis ~ gamma_peer_dis_higher*educat
  
  peer_t_lower := int_peer_t
  peer_dis_lower := int_peer_dis
  peer_p_lower := int_peer_t + int_peer_dis
  
  peer_t_higher := int_peer_t + gamma_peer_t_higher
  peer_dis_higher := int_peer_dis + gamma_peer_dis_higher
  peer_p_higher := int_peer_t + gamma_peer_t_higher + int_peer_dis + gamma_peer_dis_higher
  '
#summary of SEM fit, unweighted
fit_peer_lds_educat <- sem(peer_lds_educat, data=data, estimator = "MLR")
summary(fit_peer_lds_educat, standardized=TRUE, fit.measures=TRUE)

## weighted statistics
fit_peer_lds_educat_svy <- lavaan.survey(fit_peer_lds_educat, weight, estimator="ML")
summary(fit_peer_lds_educat_svy, standardized=TRUE, fit.measures=TRUE)

#Plot weighted
data_plot <- parameterEstimates(fit_peer_lds_educat_svy)[c(49,51,52,54),c(4:10)]
data_plot$Educat <- factor(c(0,0,1,1), levels=c(0,1), labels = c("No Highers", "Highers"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "MC"))
data_plot

ggplot(data_plot, aes(Reporter, est, colour=Educat)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4)
