################################################################################################
##### Parent-teacher discrepancy modelling for the emotional difficulties  sweep 8          ####
################################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw8-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw8clean(root)

# Survey weighting
weight <- svydesign(id=~DhPSU, strata=~DhStrat, weights=~DhWTbrth, data=data)
options(survey.lonely.psu = "adjust")

# bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)

#####################################################
##### Sample statistics sweep 8        ##############
#####################################################

## gender
summary(data$sex, na.rm=TRUE)

## tenure
summary(data$tenancy, na.rm=T)

## family composition
summary(data$family, na.rm=T)

## financial wellbeing
summary(data$finance, na.rm=T)

## maternal education
summary(data$educat, na.rm=T)
table(data$educat)

## maternal depression
mean(data$mcs12, na.rm=T)
summary(data$mcs12, na.rm=T)

#####################################
##### Metric invariance testing######
######################################

# Configural model
emo_config <- '
# latent variable definitions
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + ThSDQ08 + ThSDQ13 + ThSDQ16 + ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + MhSDQ08 + MhSDQ13 + MhSDQ16 + MhSDQ24

# variances and covariances
  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ 1
  ThSDQ13 ~ 1
  ThSDQ16 ~ 1
  ThSDQ24 ~ 1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ 1
  MhSDQ13 ~ 1
  MhSDQ16 ~ 1
  MhSDQ24 ~ 1

# intercepts
  emo_t ~ 0*1
  emo_p ~ 1
  
  emo_t ~~ 1*emo_t
  emo_p ~~ emo_p
  emo_t ~~ emo_p
'

fit_emo_config <- cfa(emo_config, data=data, estimator="MLR")
summary(fit_emo_config, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_emo_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric Invariance
emo_metr <- '
# latent variable definitions
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ 1
  ThSDQ13 ~ 1
  ThSDQ16 ~ 1
  ThSDQ24 ~ 1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ 1
  MhSDQ13 ~ 1
  MhSDQ16 ~ 1
  MhSDQ24 ~ 1

  emo_t ~ 0*1
  emo_p ~ 1
  
  emo_t ~~ 1*emo_t
  emo_p ~~ emo_p
  emo_t ~~ emo_p
'

fit_emo_metr <- cfa(emo_metr, data=data, estimator="MLR")
summary(fit_emo_metr, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_emo_metr, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_emo_config, "cfi.scaled") - fitmeasures(fit_emo_metr, "cfi.scaled") 
fitmeasures(fit_emo_config, "rmsea.scaled") - fitmeasures(fit_emo_metr, "rmsea.scaled")
fitmeasures(fit_emo_config, "srmr") - fitmeasures(fit_emo_metr, "srmr")  

# Scalar invariance
emo_scal <- '
# latent variable definitions
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ nu3*1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ nu3*1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1

  emo_t ~ 0*1
  emo_p ~ 1
  
  emo_t ~~ 1*emo_t
  emo_p ~~ emo_p
  emo_t ~~ emo_p
'

fit_emo_scal <- cfa(emo_scal, data=data, estimator="MLR")
summary(fit_emo_scal, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_emo_scal, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_emo_metr, "cfi.scaled") - fitmeasures(fit_emo_scal, "cfi.scaled") 
fitmeasures(fit_emo_metr, "rmsea.scaled") - fitmeasures(fit_emo_scal, "rmsea.scaled")
fitmeasures(fit_emo_metr, "srmr") - fitmeasures(fit_emo_scal, "srmr") 

# See which intercept to free
lavTestScore(fit_emo_scal) # p.17 to 22 is highest chi2

## Partial scalar model
emo_scal_par <- '
# latent variable definitions
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ 1
  ThSDQ13 ~ nu3*1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ 1
  MhSDQ13 ~ nu3*1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1

  emo_t ~ 0*1
  emo_p ~ 1
  
  emo_t ~~ 1*emo_t
  emo_p ~~ emo_p
  emo_t ~~ emo_p
'

fit_emo_scal_par <- cfa(emo_scal_par, data=data, estimator="MLR")
summary(fit_emo_scal_par, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_emo_scal_par, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

# differences to configural model fit
fitmeasures(fit_emo_metr, "cfi.scaled") - fitmeasures(fit_emo_scal_par, "cfi.scaled") 
fitmeasures(fit_emo_metr, "rmsea.scaled") - fitmeasures(fit_emo_scal_par, "rmsea.scaled")
fitmeasures(fit_emo_metr, "srmr") - fitmeasures(fit_emo_scal_par, "srmr")

#############################################################
####################  LDS Models    #########################
#############################################################

emo_lds <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p # discrepancy of parental reports
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ 1
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ 1
  emo_dis ~~ emo_dis
'

## Unweighted estimates
fit_emo_lds <- sem(emo_lds, data=data, estimator="MLR")
summary(fit_emo_lds, standardized=T, fit.measures=T)

## Weighting with bootstrapped errors
boot_emo1 <- lavaan.survey(fit_emo_lds, weight_boot)
summary(boot_emo1, standardized=T, fit.measures=T)

## Compare estimates by unweighted, weighted, and bootstrapped weighted
parameterEstimates(fit_emo_lds, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]
parameterEstimates(boot_emo1, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)]

## reminder: std.nox (c 13), r 33 is the Mean LDS std.
## reminder: est (c 5), r 34 is the variance.
## reminder: std.nox (c13), r 28 is the covariance std.

#################################################################################
######################### Conditional models              #######################
#################################################################################

## Conditional on the study child's biological sex, 0:male and 1: female, ref=female.

emo_lds_sex <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p # discrepancy of parental reports
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ int_emo_t*1 + gamma_emo_t_female*sex
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ int_emo_dis*1
  emo_dis ~~ emo_dis
  
  emo_dis ~ gamma_emo_dis_female*sex
  
  emo_t_male := int_emo_t
  emo_dis_male  := int_emo_dis
  emo_p_male := int_emo_t + int_emo_dis
  
  emo_t_female := int_emo_t + gamma_emo_t_female
  emo_dis_female := int_emo_dis + gamma_emo_dis_female
  emo_p_female := int_emo_t + gamma_emo_t_female + int_emo_dis + gamma_emo_dis_female
  '

#summary of SEM fit, unweighted
fit_emo_lds_sex <- sem(emo_lds_sex, data=data, estimator = "MLR")
summary(fit_emo_lds_sex, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_sex <- lavaan.survey(fit_emo_lds_sex, weight_boot)
summary(boot_emo_sex, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_emo_lds_sex)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_emo_sex)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_sex)[c(49,51,52,54),c(4:10)]
data_plot$Gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Boy", "Girl"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

a <- ggplot(data_plot, aes(Reporter, est, colour=Gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) +  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent')
  )


## LDS conditional on parental mental health (ref = no mental health issues)

emo_lds_depress <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ int_emo_t*1 + gamma_emo_t_high*mcs12
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ int_emo_dis*1
  emo_dis ~~ emo_dis
  
  emo_dis ~ gamma_emo_dis_high*mcs12
  
  emo_t_low := int_emo_t
  emo_dis_low := int_emo_dis
  emo_p_low := int_emo_t + int_emo_dis
  
  emo_t_high := int_emo_t + gamma_emo_t_high
  emo_dis_high := int_emo_dis + gamma_emo_dis_high
  emo_p_high := int_emo_t + gamma_emo_t_high + int_emo_dis + gamma_emo_dis_high
  '
#summary of SEM fit, unweighted
fit_emo_lds_depress <- sem(emo_lds_depress, data=data, estimator = "MLR")
summary(fit_emo_lds_depress, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_depress <- lavaan.survey(fit_emo_lds_depress, weight_boot)
summary(boot_emo_depress, standardized=T, fit.measures=T)

## Comparing parameter estimates
print("Compare estimates conditional on maternal depression")
print("unweighted")
parameterEstimates(fit_emo_lds_depress)[c(49,51,52,54),c(4:10)]
print("weighted")
parameterEstimates(boot_emo_depress)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_depress)[c(49,51,52,54),c(4:10)]
data_plot$MCS <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Below average", "Average"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

b <- ggplot(data_plot, aes(Reporter, est, colour=MCS)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4)  +  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent')
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditional on family tenure (ref = own home)

emo_lds_tenancy <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p # discrepancy of parental reports
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ int_emo_t*1 + gamma_emo_t_own*tenancy
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ int_emo_dis*1
  emo_dis ~~ emo_dis
  
  emo_dis ~ gamma_emo_dis_own*tenancy
  
  emo_t_rent := int_emo_t
  emo_dis_rent := int_emo_dis
  emo_p_rent := int_emo_t + int_emo_dis
  
  emo_t_own := int_emo_t + gamma_emo_t_own
  emo_dis_own := int_emo_dis + gamma_emo_dis_own
  emo_p_own := int_emo_t + gamma_emo_t_own + int_emo_dis + gamma_emo_dis_own
  '
#summary of SEM fit, unweighted
fit_emo_lds_tenancy <- sem(emo_lds_tenancy, data=data, estimator = "MLR")
summary(fit_emo_lds_tenancy, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_tenancy <- lavaan.survey(fit_emo_lds_tenancy, weight_boot)
summary(boot_emo_tenancy, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_emo_lds_tenancy)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_emo_tenancy)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_tenancy)[c(49,51,52,54),c(4:10)]
data_plot$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

c <- ggplot(data_plot, aes(Reporter, est, colour=Tenancy)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4)  +  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Conditional on maternal educational attainment (ref = Scottish highers or above)

emo_lds_educat <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p # discrepancy of parental reports
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ int_emo_t*1 + gamma_emo_t_higher*educat
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ int_emo_dis*1
  emo_dis ~~ emo_dis
  
  emo_dis ~ gamma_emo_dis_higher*educat
  
  emo_t_lower := int_emo_t
  emo_dis_lower := int_emo_dis
  emo_p_lower := int_emo_t + int_emo_dis
  
  emo_t_higher := int_emo_t + gamma_emo_t_higher
  emo_dis_higher := int_emo_dis + gamma_emo_dis_higher
  emo_p_higher := int_emo_t + gamma_emo_t_higher + int_emo_dis + gamma_emo_dis_higher
  '

#summary of SEM fit, unweighted
fit_emo_lds_educat <- sem(emo_lds_educat, data=data, estimator = "MLR")
summary(fit_emo_lds_educat, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_educat <- lavaan.survey(fit_emo_lds_educat, weight_boot)
summary(boot_emo_educat, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_emo_lds_educat)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_emo_educat)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_educat)[c(49,51,52,54),c(4:10)]
data_plot$Education <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Below Highers", "Highers"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

d <- ggplot(data_plot, aes(Reporter, est, colour=Education)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent')
  )

## Conditional on parents reporting of subjective financial wellbeing (ref = managing well)
emo_lds_finance <- '
  emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
  emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

  ThSDQ03 ~~ MhSDQ03
  ThSDQ08 ~~ MhSDQ08
  ThSDQ13 ~~ MhSDQ13
  ThSDQ16 ~~ MhSDQ16
  ThSDQ24 ~~ MhSDQ24
  
  ThSDQ03 ~ nu1*1
  ThSDQ08 ~ nu2*1
  ThSDQ13 ~ 1
  ThSDQ16 ~ nu4*1
  ThSDQ24 ~ nu5*1
  
  MhSDQ03 ~ nu1*1
  MhSDQ08 ~ nu2*1
  MhSDQ13 ~ 1
  MhSDQ16 ~ nu4*1
  MhSDQ24 ~ nu5*1
  
  # LDS model
  
  emo_dis =~ 1*emo_p # discrepancy of parental reports
  emo_p ~ 1*emo_t
  emo_dis ~~ emo_t
  
  emo_t ~~ emo_t
  emo_t ~ int_emo_t*1 + gamma_emo_t_well*finance
  
  emo_p ~~ 0*emo_p
  emo_p ~ 0*1
  
  emo_dis ~ int_emo_dis*1
  emo_dis ~~ emo_dis
  
  emo_dis ~ gamma_emo_dis_well*finance
  
  emo_t_other := int_emo_t
  emo_dis_other := int_emo_dis
  emo_p_other := int_emo_t + int_emo_dis
  
  emo_t_well := int_emo_t + gamma_emo_t_well
  emo_dis_well := int_emo_dis + gamma_emo_dis_well
  emo_p_well := int_emo_t + gamma_emo_t_well + int_emo_dis + gamma_emo_dis_well
  '

#summary of SEM fit, unweighted
fit_emo_lds_finance <- sem(emo_lds_finance, data=data, estimator = "MLR")
summary(fit_emo_lds_finance, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_finance <- lavaan.survey(fit_emo_lds_finance, weight_boot)
summary(boot_emo_finance, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_emo_lds_finance)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_emo_finance)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_finance)[c(49,51,52,54),c(4:10)]
data_plot$Finances <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Other", "Managing well"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

e <- ggplot(data_plot, aes(Reporter, est, colour=Finances)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4)  + 
  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent')
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## Condition on family composition (ref = couple household)

emo_lds_family <- '
emo_t =~ NA*ThSDQ03 + lambda1*ThSDQ03 + lambda2*ThSDQ08 + lambda3*ThSDQ13 + lambda4*ThSDQ16 + lambda5*ThSDQ24
emo_p =~ NA*MhSDQ03 + lambda1*MhSDQ03 + lambda2*MhSDQ08 + lambda3*MhSDQ13 + lambda4*MhSDQ16 + lambda5*MhSDQ24

ThSDQ03 ~~ MhSDQ03
ThSDQ08 ~~ MhSDQ08
ThSDQ13 ~~ MhSDQ13
ThSDQ16 ~~ MhSDQ16
ThSDQ24 ~~ MhSDQ24
  
ThSDQ03 ~ nu1*1
ThSDQ08 ~ nu2*1
ThSDQ13 ~ 1
ThSDQ16 ~ nu4*1
ThSDQ24 ~ nu5*1
  
MhSDQ03 ~ nu1*1
MhSDQ08 ~ nu2*1
MhSDQ13 ~ 1
MhSDQ16 ~ nu4*1
MhSDQ24 ~ nu5*1
  
  # LDS model
  
emo_dis =~ 1*emo_p # discrepancy of parental reports
emo_p ~ 1*emo_t
emo_dis ~~ emo_t

emo_t ~~ emo_t
emo_t ~ int_emo_t*1 + gamma_emo_t_couple*family
  
emo_p ~~ 0*emo_p
emo_p ~ 0*1
  
emo_dis ~ int_emo_dis*1
emo_dis ~~ emo_dis
  
emo_dis ~ gamma_emo_dis_couple*family
  
emo_t_single := int_emo_t
emo_dis_single := int_emo_dis
emo_p_single := int_emo_t + int_emo_dis
  
emo_t_couple := int_emo_t + gamma_emo_t_couple
emo_dis_couple := int_emo_dis + gamma_emo_dis_couple
emo_p_couple := int_emo_t + gamma_emo_t_couple + int_emo_dis + gamma_emo_dis_couple
'

#summary of SEM fit, unweighted
fit_emo_lds_family <- sem(emo_lds_family, data=data, estimator = "MLR")
summary(fit_emo_lds_family, standardized=TRUE, fit.measures=TRUE)

## Weighted with bootstrapping
boot_emo_family <- lavaan.survey(fit_emo_lds_family, weight_boot)
summary(boot_emo_family, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_emo_lds_family)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_emo_family)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot <- parameterEstimates(boot_emo_family)[c(49,51,52,54),c(4:10)]
data_plot$Family <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Single parent", "Couple"))
data_plot$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Teacher", "Parent"))
data_plot

f <- ggplot(data_plot, aes(Reporter, est, colour=Family)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Emotion factor")+
  ylim(-0.4, +0.4)  + 
  theme(
    panel.background = element_rect(fill='transparent'),
    plot.background = element_rect(fill='transparent', color=NA),
    panel.grid.major = element_blank(),
    panel.grid.minor = element_blank(),
    legend.background = element_rect(fill='transparent'),
    legend.box.background = element_rect(fill='transparent')
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )



#####################################
##### Combine plots and export ######
######################################

library(cowplot)

five <- cowplot::plot_grid(a + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              b + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              c + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              d + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              e + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              f + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                              ncol = 3, nrow = 2,
                              rel_heights = c(1, 1),
                              rel_widths = c(1, 1, 1)) + theme(plot.margin = margin(20, 20, 10, 10))

## save plot as png
ggsave(file.path("C:", "Users", "s1769862", "Desktop", "LDS-plots", "emo-10-conditional.png"), five, width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as Rdata
saveRDS(five, file.path("C:", "Users", "s1769862", "OneDrive - University of Edinburgh", "SDQ-paper", "plotdata", "plot5.Rdata"))

