#############################################################################################
##### Parent-adolescent discrepancy modelling for the pro-social behaviour scale sweep 10####
#############################################################################################

### Packages
library(dplyr)
library(lavaan)
library(lavaan.survey)
library(ggplot2)
library(ggthemes)
library(cowplot)

## Import and clean data
source("M:/A-matter-of-perspective-code/sw10-clean-function.R")
root <- "M:/GUS Data Files/GUStabs" # replace with your file system location
data <- sw10clean(root)

# Survey weighting
options(survey.lonely.psu = "adjust")
weight <- svydesign(id=~Djpsu, strata=~Djstrata, weights=~Djwtbrthf, data=data)

## bootstrapped weight
weight_boot<-as.svrepdesign(weight, type='bootstrap', replicates=500)

#####################################
##### Metric invariance testing######
######################################

## Configural invariance model

pro_config <- '
# latent variable definitions
  pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + CjSDQ04 + CjSDQ09 + CjSDQ17 + CjSDQ20
  pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + MjSDQ04 + MjSDQ09 + MjSDQ17 + MjSDQ20

# variances and covariances
  CjSDQ01 ~~ MjSDQ01
  CjSDQ04 ~~ MjSDQ04
  CjSDQ09 ~~ MjSDQ09
  CjSDQ17 ~~ MjSDQ17
  CjSDQ20 ~~ MjSDQ20
  
  CjSDQ01 ~ nu1*1
  CjSDQ04 ~ 1
  CjSDQ09 ~ 1
  CjSDQ17 ~ 1
  CjSDQ20 ~ 1
  
  MjSDQ01 ~ nu1*1
  MjSDQ04 ~ 1
  MjSDQ09 ~ 1
  MjSDQ17 ~ 1
  MjSDQ20 ~ 1

# intercepts 
  pro_c ~ 0*1
  pro_p ~ 1
  
  pro_c ~~ 1*pro_c
  pro_p ~~ pro_p
  pro_c ~~ pro_p
'

fit_pro_config <- cfa(pro_config, data=data, estimator="MLR")
summary(fit_pro_config, standardized=TRUE, fit.measures=TRUE)
fitmeasures(fit_pro_config, fit.measures = c("cfi.scaled", "rmsea.scaled", "srmr"))

## Metric model
pro_metr <- ' 
pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20

CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20

CjSDQ01 ~ nu1*1
CjSDQ04 ~ 1
CjSDQ09 ~ 1
CjSDQ17 ~ 1
CjSDQ20 ~ 1

MjSDQ01 ~ nu1*1
MjSDQ04 ~ 1
MjSDQ09 ~ 1
MjSDQ17 ~ 1
MjSDQ20 ~ 1

pro_c ~ 0*1
pro_p ~ 1

pro_c ~~ 1*pro_c
pro_p ~~ pro_p
pro_c ~~ pro_p
'

fit_pro_metr <- cfa(pro_metr, data=data, estimator="MLR")
summary(fit_pro_metr, standardized=T, fit.measures=T)
fitMeasures(fit_pro_metr, fit.measures = c("cfi.scaled","rmsea.scaled", "srmr"))

# fit differences
fitmeasures(fit_pro_config, "cfi.scaled") - fitmeasures(fit_pro_metr, "cfi.scaled") 
fitmeasures(fit_pro_config, "rmsea.scaled") - fitmeasures(fit_pro_metr, "rmsea.scaled")
fitmeasures(fit_pro_config, "srmr") - fitmeasures(fit_pro_metr, "srmr")  

### Scalar model 
pro_scal <- ' 
  pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
  pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
  CjSDQ01 ~~ MjSDQ01
  CjSDQ04 ~~ MjSDQ04
  CjSDQ09 ~~ MjSDQ09
  CjSDQ17 ~~ MjSDQ17
  CjSDQ20 ~~ MjSDQ20
  
  CjSDQ01 ~ nu1*1
  CjSDQ04 ~ nu2*1
  CjSDQ09 ~ nu3*1
  CjSDQ17 ~ nu4*1
  CjSDQ20 ~ nu5*1
  
  MjSDQ01 ~ nu1*1
  MjSDQ04 ~ nu2*1
  MjSDQ09 ~ nu3*1
  MjSDQ17 ~ nu4*1
  MjSDQ20 ~ nu5*1
  
  pro_c ~ 0*1
  pro_p ~ 1
  
  pro_c ~~ 1*pro_c
  pro_p ~~ pro_p
  pro_c ~~ pro_p
'

fit_pro_scal <- cfa(pro_scal, data=data, estimator="MLR")
summary(fit_pro_scal, standardized=T, fit.measures=T)
fitMeasures(fit_pro_scal, fit.measures = c("cfi.scaled","rmsea.scaled", "srmr"))

# fit differences

fitmeasures(fit_pro_metr, "cfi.scaled") - fitmeasures(fit_pro_scal, "cfi.scaled") 
fitmeasures(fit_pro_metr, "rmsea.scaled") - fitmeasures(fit_pro_scal, "rmsea.scaled")
fitmeasures(fit_pro_metr, "srmr") - fitmeasures(fit_pro_scal, "srmr")

### Change in fit exceeded acceptable levels - try for partial scalar model

lavTestScore(fit_pro_scal, standardized = epc)

# For scalar partial the intercepts for item 1 (I am nice) was freed due to highest x2 value in lav test

### Partial scalar configuration model 
pro_scal_par <- ' 
  pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
  pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
  CjSDQ01 ~~ MjSDQ01
  CjSDQ04 ~~ MjSDQ04
  CjSDQ09 ~~ MjSDQ09
  CjSDQ17 ~~ MjSDQ17
  CjSDQ20 ~~ MjSDQ20
  
  CjSDQ01 ~ 1
  CjSDQ04 ~ nu2*1
  CjSDQ09 ~ nu3*1
  CjSDQ17 ~ nu4*1
  CjSDQ20 ~ nu5*1
  
  MjSDQ01 ~ 1
  MjSDQ04 ~ nu2*1
  MjSDQ09 ~ nu3*1
  MjSDQ17 ~ nu4*1
  MjSDQ20 ~ nu5*1
  
  pro_c ~ 0*1
  pro_p ~ 1
  
  pro_c ~~ 1*pro_c
  pro_p ~~ pro_p
  pro_c ~~ pro_p '

fit_pro_scal_par <- cfa(pro_scal_par, data=data, estimator="MLR")
summary(fit_pro_scal_par, standardized=T, fit.measures=T)
fitMeasures(fit_pro_scal_par, fit.measures = c("cfi.scaled","rmsea.scaled", "srmr"))

# fit differences

fitmeasures(fit_pro_metr, "cfi.scaled") - fitmeasures(fit_pro_scal_par, "cfi.scaled") 
fitmeasures(fit_pro_metr, "rmsea.scaled") - fitmeasures(fit_pro_scal_par, "rmsea.scaled")
fitmeasures(fit_pro_metr, "srmr") - fitmeasures(fit_pro_scal_par, "srmr")

######################################
######## LDS models ##################
######################################

### Latent difference score model
pro_lds <-
'
pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1

  
pro_dis =~ 1*pro_p # latent factor definition
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ 1
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ 1
pro_dis ~~ pro_dis
'

## Unweighted summary
fit_pro_lds <- sem(pro_lds, data=data, estimator="MLR")
summary(fit_pro_lds, standardized=T, fit.measures=T)

## Weighting with bootstrapped errors
boot_pro1 <- lavaan.survey(fit_pro_lds, weight_boot)
summary(boot_pro1, standardized=T, fit.measures=T)

# Compare LDS estimates
parameterEstimates(fit_pro_lds, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)] # unweigthed
parameterEstimates(boot_pro1, standardized=T)[c(28, 33, 34), c(1, 5, 8, 9, 10, 13)] # weighted


## reminder: std.nox (c 13), r 33 is the Mean LDS std.
## reminder: est (c 5), r 34 is the variance.
## reminder: std.nox (c13), r 28 is the covariance std.

#####################################
##### Conditional LDS models   ######
######################################

### LDS model conditional on adolescent gender
##Reminder: CjGenId 0 = male, 1 = female

pro_lds_sex <- ' 

  # Measurement model, partial scalar

pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
    # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy factor definition
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_girls*gender # conditioning the child on gender binary
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning discrepancy factor on gender binary
  
pro_dis ~ gamma_pro_dis_girls*gender
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_boys := int_pro_c # boys are intercept of child score
pro_dis_boys := int_pro_dis # is defined as
pro_p_boys := int_pro_c + int_pro_dis
    
pro_c_girls := int_pro_c + gamma_pro_c_girls # girls are boys + impact of binary positive
pro_dis_girls := int_pro_dis + gamma_pro_dis_girls
pro_p_girls := int_pro_c + gamma_pro_c_girls + int_pro_dis + gamma_pro_dis_girls
'

######## Unweighed estimates #######
fit_pro_lds_sex <- sem(pro_lds_sex, data=data, estimator="MLR")
summary(fit_pro_lds_sex, standardized=T, fit.measures=T)

###### Weighted with bootstrapping ######
boot_pro_sex <- lavaan.survey(fit_pro_lds_sex, weight_boot)
summary(boot_pro_sex, standardized=T, fit.measures=T)

###### Comparing parameter estimates ######
parameterEstimates(fit_pro_lds_sex)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_sex)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_a <- parameterEstimates(boot_pro_sex)[c(49,51,52,54),c(4:10)]
data_plot_a$Gender <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Boy", "Girl"))
data_plot_a$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_a

a <- ggplot(data_plot_a, aes(Reporter, est, colour=Gender)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )


## Parental experiences of depression

pro_lds_distress <- ' 

  # Measurement model, partial scalar
  
pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
  # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy is measured by 1 times parent report
pro_p ~ 1*pro_c # parent report is regression on 1 times child report
pro_dis ~~ pro_c # the covariance of discerpancy and child report
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_no*depress # conditioning the "intercept"/parent distress on reference of never depression=1
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning LDS on parent no depression
  
pro_dis ~ gamma_pro_dis_no*depress
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_yes := int_pro_c
pro_dis_yes := int_pro_dis
pro_p_yes := int_pro_c + int_pro_dis
    
pro_c_no := int_pro_c + gamma_pro_c_no
pro_dis_no := int_pro_dis + gamma_pro_dis_no
pro_p_no := int_pro_c + gamma_pro_c_no + int_pro_dis + gamma_pro_dis_no
'

# unweighted
fit_pro_lds_distress <- sem(pro_lds_distress, data=data, estimator="MLR")
summary(fit_pro_lds_distress, standardized=T, fit.measures=T)

## Weighted with bootstrapping
boot_pro_distress <- lavaan.survey(fit_pro_lds_distress, weight_boot)
summary(boot_pro_distress, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_distress)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_distress)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_b <- parameterEstimates(boot_pro_distress)[c(49,51,52,54),c(4:10)]
data_plot_b$Depression <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Yes", "No"))
data_plot_b$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_b

b <- ggplot(data_plot_b, aes(Reporter, est, colour=Depression)) +
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) + theme_hc()+ scale_colour_hc()+
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  ) +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

 
## Conditional on housing tenure

pro_lds_tenancy <- ' 

  # Measurement model, partial scalar
  
pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
  # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy of parents reports
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_own*tenancy # conditioning the "intercept"/tenancy type
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning LDS on tenancy type "Own home"
  
pro_dis ~ gamma_pro_dis_own*tenancy
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_rent := int_pro_c
pro_dis_rent := int_pro_dis
pro_p_rent := int_pro_c + int_pro_dis
    
pro_c_own := int_pro_c + gamma_pro_c_own
pro_dis_own := int_pro_dis + gamma_pro_dis_own
pro_p_own := int_pro_c + gamma_pro_c_own + int_pro_dis + gamma_pro_dis_own
'

# Unweighted summary
fit_pro_lds_tenancy <- sem(pro_lds_tenancy, data=data, estimator="MLR")
summary(fit_pro_lds_tenancy, standardized=T, fit.measures=T)

# Weighted with bootstrapping
boot_pro_tenancy <- lavaan.survey(fit_pro_lds_tenancy, weight_boot)
summary(boot_pro_tenancy, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_tenancy)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_tenancy)[c(49,51,52,54),c(4:10)]


#Plot weighted
data_plot_c <- parameterEstimates(boot_pro_tenancy)[c(49,51,52,54),c(4:10)]
data_plot_c$Tenancy <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Rent", "Own home"))
data_plot_c$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_c

c <- ggplot(data_plot_c, aes(Reporter, est, colour=Tenancy)) +
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) + theme_hc() + scale_colour_hc()+
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  ) +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )


## Family composition

pro_lds_family <-'# Measurement model, partial scalar

pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
    # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy of parents reports
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_couple*family # conditioning the "intercept"/child
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning LDS on family composition
  
pro_dis ~ gamma_pro_dis_couple*family
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_single := int_pro_c
pro_dis_single := int_pro_dis
pro_p_single := int_pro_c + int_pro_dis
    
pro_c_couple := int_pro_c + gamma_pro_c_couple
pro_dis_couple := int_pro_dis + gamma_pro_dis_couple
pro_p_couple := int_pro_c + gamma_pro_c_couple + int_pro_dis + gamma_pro_dis_couple
'

# Unweighted
fit_pro_lds_family <- sem(pro_lds_family, data=data, estimator="MLR")
summary(fit_pro_lds_family, standardized=T, fit.measures=T)

## Weighted with bootstrapping
boot_pro_family <- lavaan.survey(fit_pro_lds_family, weight_boot)
summary(boot_pro_family, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_family)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_family)[c(49,51,52,54),c(4:10)]


#Plot weighted
data_plot_d <- parameterEstimates(boot_pro_family)[c(49,51,52,54),c(4:10)]
data_plot_d$Family <- factor(c(0,0,1,1), levels=c(0,1), labels = c("Single parent", "Couple"))
data_plot_d$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_d

d <- ggplot(data_plot_d, aes(Reporter, est, colour=Family)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) +
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )


## Language of the home

pro_lds_lang <- ' 

  # Measurement model, partial scalar

pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
    # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy of parents reports
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_english*lang # conditioning the "intercept"/child
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning LDS on household language
  
pro_dis ~ gamma_pro_dis_english*lang
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_other := int_pro_c
pro_dis_other := int_pro_dis
pro_p_other := int_pro_c + int_pro_dis
    
pro_c_english := int_pro_c + gamma_pro_c_english
pro_dis_english := int_pro_dis + gamma_pro_dis_english
pro_p_english := int_pro_c + gamma_pro_c_english + int_pro_dis + gamma_pro_dis_english
'

# unweighyted
fit_pro_lds_lang <- sem(pro_lds_lang, data=data, estimator="MLR")
summary(fit_pro_lds_lang, standardized=T, fit.measures=T)

# weighted with bootstrapping
boot_pro_lang <- lavaan.survey(fit_pro_lds_lang, weight_boot)
summary(boot_pro_lang, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_lang)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_lang)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_e <- parameterEstimates(boot_pro_lang)[c(49,51,52,54),c(4:10)]
data_plot_e$Language <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Only English"))
data_plot_e$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_e

e <- ggplot(data_plot_e, aes(Reporter, est, colour=Language)) + theme_hc()+ scale_colour_hc()+
geom_point(size=2) +
geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
xlab("") + 
ylab("Pro-social factor")+
ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

## subjective household finances

pro_lds_finance <- ' 

  # Measurement model, partial scalar

pro_c =~ NA*CjSDQ01 + lambda1*CjSDQ01 + lambda2*CjSDQ04 + lambda3*CjSDQ09 + lambda4*CjSDQ17 + lambda5*CjSDQ20
pro_p =~ NA*MjSDQ01 + lambda1*MjSDQ01 + lambda2*MjSDQ04 + lambda3*MjSDQ09 + lambda4*MjSDQ17 + lambda5*MjSDQ20
  
CjSDQ01 ~~ MjSDQ01
CjSDQ04 ~~ MjSDQ04
CjSDQ09 ~~ MjSDQ09
CjSDQ17 ~~ MjSDQ17
CjSDQ20 ~~ MjSDQ20
  
CjSDQ01 ~ 1
CjSDQ04 ~ nu2*1
CjSDQ09 ~ nu3*1
CjSDQ17 ~ nu4*1
CjSDQ20 ~ nu5*1
  
MjSDQ01 ~ 1
MjSDQ04 ~ nu2*1
MjSDQ09 ~ nu3*1
MjSDQ17 ~ nu4*1
MjSDQ20 ~ nu5*1
    
    # LDS model 
  
pro_dis =~ 1*pro_p # Discrepancy of parents reports
pro_p ~ 1*pro_c
pro_dis ~~ pro_c
    
pro_c ~~ pro_c
pro_c ~ int_pro_c*1 + gamma_pro_c_well*finance # conditioning the "intercept"/child
    
pro_p ~~ 0*pro_p
pro_p ~ 0*1
    
pro_dis ~ int_pro_dis*1
pro_dis ~~ pro_dis
    
  # Conditioning LDS on financial health
  
pro_dis ~ gamma_pro_dis_well*finance
    
  # Simple intercepts and slopes (discrepancies)
  
pro_c_other := int_pro_c
pro_dis_other := int_pro_dis
pro_p_other := int_pro_c + int_pro_dis
    
pro_c_well := int_pro_c + gamma_pro_c_well
pro_dis_well := int_pro_dis + gamma_pro_dis_well
pro_p_well := int_pro_c + gamma_pro_c_well + int_pro_dis + gamma_pro_dis_well
'
# unweighted
fit_pro_lds_finance <- sem(pro_lds_finance, data=data, estimator="MLR")
summary(fit_pro_lds_finance, standardized=T, fit.measures=T)

## weighted with bootstrapping
boot_pro_finance <- lavaan.survey(fit_pro_lds_finance, weight_boot)
summary(boot_pro_finance, standardized=T, fit.measures=T)

## Comparing parameter estimates
parameterEstimates(fit_pro_lds_finance)[c(49,51,52,54),c(4:10)]
parameterEstimates(boot_pro_finance)[c(49,51,52,54),c(4:10)]

#Plot weighted
data_plot_f <- parameterEstimates(boot_pro_finance)[c(49,51,52,54),c(4:10)]
data_plot_f$Finances <- factor(c(0,0,1,1), levels=c(0,1), labels=c("Other", "Managing well"))
data_plot_f$Reporter <- factor(c(1,0,1,0), levels=c(1,0), labels=c("Adolescent", "Parent"))
data_plot_f

f <- ggplot(data_plot_f, aes(Reporter, est, colour=Finances)) + theme_hc()+ scale_colour_hc()+
  geom_point(size=2) +
  geom_errorbar(aes(ymin = ci.lower, ymax = ci.upper), width=.2, linewidth=.3) +
  xlab("") + 
  ylab("Pro-social factor")+
  ylim(-0.4, +0.4) + 
  theme(
    panel.background = element_rect(fill='transparent'), #transparent panel bg
    plot.background = element_rect(fill='transparent', color=NA), #transparent plot bg
    panel.grid.major = element_blank(), #remove major gridlines
    panel.grid.minor = element_blank(), #remove minor gridlines
    legend.background = element_rect(fill='transparent'), #transparent legend bg
    legend.box.background = element_rect(fill='transparent') #transparent legend panel
  )  +
  theme(axis.text.y=element_blank(),
        axis.ticks.y=element_blank() 
  )

#####################################
##### Combine plots and export ######
######################################

library(cowplot)

one <- cowplot::plot_grid(a + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          b + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          c + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          d + theme(axis.title.y = element_text(size = 12), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          e + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          f + theme(axis.title.y = element_blank(),  axis.text.y = element_blank(), legend.title = element_text(size=10), legend.text = element_text(size = 8), legend.position = "bottom", legend.key.size = unit(0.5, "cm"), plot.margin = margin(0, 0, 0, 0)),
                          ncol = 3, nrow = 2,
                          rel_heights = c(1, 1),
                          rel_widths = c(1, 1, 1)) + theme(plot.margin = margin(20, 20, 10, 10))
## save plot as png
ggsave(file.path("C", "Users", "s1769862", "Desktop", "LDS-plots", "social-14-conditional.png"), one,  width=9, height=7, dpi=500, device=png, bg='transparent')

## save plot as Rdata
saveRDS(one, file.path("C:", "Users", "s1769862", "OneDrive - University of Edinburgh", "SDQ-paper", "plotdata", "plot1.Rdata"))
