# Scale direction exploration

# How scale direction effects responses?
# What is the moderating effect of scale and question characteristics?

# Design:
# - positive to negative vs. low to high
# - attitudinal with agreement vs. behavior with frequency
# - 5 vs. 7 point scale
# - full labels vs. endpoints
# -  2 (scale direction)*2 (question type)*2 (scale length)*2 (scale labeling)
# - same design in 2 waves except
#   - half get the same scale direction and have change

# code focuses on analysis


# Admin ------------

# install packages
# install.packages("tidyverse", dep = T)
# install.packages("haven", dep = T)
# install.packages("laveen", dep = T)
# install.packages("semTools", dep = T)


library(tidyverse)
library(haven)
library(lavaan)
library(semTools)
library(viridis)
library(blavaan)

# load data
load("./data/clean/clean_data.RData")


# analysis ----------

var_labels <- c("Automatic", "Rush", "Absorbed", "Several",
                "Wanders", "Distracted",
                "Complicated", "Understanding", "Care", "Say",
                "Vote", "Journalists", "Officials", "Choice",
                "Rich")

var_names <- str_c("att_", c(str_c("0", 1:9), 10:15))




## means comparison --------------

# wave 1 - by question
dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  rename_at(var_names, ~str_remove(., "_")) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T),
                       se = ~sd(., na.rm = T)/sqrt(n()))) %>%
  pivot_longer(
    cols = 5:34,
    values_to = "value",
    names_to = c("var", "stats"),
    names_pattern = "(.+)_(.+)",
  ) %>%
  pivot_wider(
    names_from = stats,
    values_from = value) %>%
  mutate(variable = factor(var, labels = var_labels) %>% fct_rev(),
         lci = mean - (1.96 * se),
         uci = mean + (1.96 * se)) %>%
  ggplot(aes(mean, exp_pos_fct,
             alpha = exp_att_fct,
             color = exp_five_fct,
             shape = exp_full_fct,
             linetype = exp_full_fct)) +
  geom_pointrange(aes(xmin = lci, xmax = uci),
                  position = position_dodge(-0.5),
                  line_width = 1, size = 0.15) +
  facet_wrap(~variable, scales = "free", ncol =3) +
  theme_bw() +
  scale_alpha_discrete(range=c(0.4, 0.9)) +
  labs(y = "",
       x = "Mean and 95% confidence intervals",
       alpha = "",
       color = "",
       shape = "",
       linetype = "")



ggsave("./output/fig/mean_dif_w1.png", dpi = 400, width = 8, height = 6.5)



# wave 1 - overall
means_w1 <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  rename_at(var_names, ~str_remove(., "_")) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T))) %>%
  pivot_longer(
    cols = 5:19,
    values_to = "value",
    names_to = c("var", "stats"),
    names_pattern = "(.+)_(.+)",
  ) %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean = mean(value))

means_w1 %>%
  ggplot(aes(mean, exp_pos_fct,
             alpha = exp_att_fct,
             color = exp_five_fct,
             shape = exp_full_fct)) +
  geom_point(position = position_dodge(-0.5), size = 2) +
  theme_bw() +
  scale_alpha_discrete(range=c(0.4, 0.9)) +
  labs(y = "",
       x = "Means of all the questions",
       alpha = "",
       color = "",
       shape = "")

ggsave("./output/fig/means_w1.png", dpi = 400, height = 3, width = 6)









dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T))) %>%
  gather(key = key, value = value,
         -exp_pos_fct, -exp_att_fct, -exp_five_fct, -exp_full_fct) %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)

# overall average
dfw1_2 %>%
  group_by(exp_pos_fct) %>%
  select(all_of(var_names)) %>%
  summarise_all(~mean(., na.rm = T)) %>%
  gather(key = key, value = value,
         -exp_pos_fct) %>%
  group_by(exp_pos_fct) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)

count(dfw1_2, exp_five, att_01)


# t.tests
mean_dif_w1 <- map(var_names, function(x) {
  t.test(dfw1_2[[x]] ~ dfw1_2$exp_pos_fct)
})


cbind(map_dbl(mean_dif_w1, function(x) x$p.value) %>% round(2),
      map_df(mean_dif_w1, function(x) x$estimate)) %>%
  as_tibble() %>%
  setNames(c("Pvalue", "Ascending", "Descending")) %>%
  mutate(Variable = var_labels,
         'Mean difference' = `Ascending` - `Descending` %>%
           round(3)) %>%
  select(Variable, `Ascending`, `Descending`, `Mean difference`,
         `Pvalue`) %>% view()

# wave 2

dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  rename_at(var_names, ~str_remove(., "_")) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T),
                       se = ~sd(., na.rm = T)/sqrt(n()))) %>%
  pivot_longer(
    cols = 5:34,
    values_to = "value",
    names_to = c("var", "stats"),
    names_pattern = "(.+)_(.+)",
  ) %>%
  pivot_wider(
    names_from = stats,
    values_from = value) %>%
  mutate(variable = factor(var, labels = var_labels) %>% fct_rev(),
         lci = mean - (1.96 * se),
         uci = mean + (1.96 * se)) %>%
  ggplot(aes(mean, exp_pos_fct,
             alpha = exp_att_fct,
             color = exp_five_fct,
             shape = exp_full_fct,
             linetype = exp_full_fct)) +
  geom_pointrange(aes(xmin = lci, xmax = uci),
                  position = position_dodge(-0.5),
                  line_width = 1, size = 0.15) +
  facet_wrap(~variable, scales = "free", ncol =3) +
  theme_bw() +
  scale_alpha_discrete(range=c(0.4, 0.9)) +
  labs(y = "",
       x = "Mean and 95% confidence intervals",
       alpha = "",
       color = "",
       shape = "",
       linetype = "")



ggsave("./output/fig/mean_dif_w2.png", dpi = 400, width = 8, height = 6.5)


dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T))) %>%
  gather(key = key, value = value,
         -exp_pos_fct, -exp_att_fct, -exp_five_fct, -exp_full_fct) %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)


# wave 1 - overall
means_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  select(all_of(var_names)) %>%
  rename_at(var_names, ~str_remove(., "_")) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T))) %>%
  pivot_longer(
    cols = 5:19,
    values_to = "value",
    names_to = c("var", "stats"),
    names_pattern = "(.+)_(.+)",
  ) %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean = mean(value))

means_w2 %>%
  ggplot(aes(mean, exp_pos_fct,
             alpha = exp_att_fct,
             color = exp_five_fct,
             shape = exp_full_fct)) +
  geom_point(position = position_dodge(-0.5), size = 2) +
  theme_bw() +
  scale_alpha_discrete(range=c(0.4, 0.9)) +
  labs(y = "",
       x = "Means of all the questions",
       alpha = "",
       color = "",
       shape = "")

ggsave("./output/fig/means_w2.png", dpi = 400, height = 3, width = 6)





# overall average
dfw2_2 %>%
  group_by(exp_pos_fct) %>%
  select(all_of(var_names)) %>%
  summarise_all(~mean(., na.rm = T)) %>%
  gather(key = key, value = value,
         -exp_pos_fct) %>%
  group_by(exp_pos_fct) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)

count(dfw2_2, exp_five, att_01)

# t.tests
mean_dif_w2 <- map(var_names, function(x) {
  t.test(dfw2_2[[x]] ~ dfw2_2$exp_pos_fct)
})


cbind(map_dbl(mean_dif_w2, function(x) x$p.value) %>% round(2),
      map_df(mean_dif_w2, function(x) x$estimate)) %>%
  as_tibble() %>%
  setNames(c("Pvalue", "Ascending", "Descending")) %>%
  mutate(Variable = var_labels,
         'Mean difference' = `Ascending` - `Descending` %>%
           round(3)) %>%
  select(Variable, `Ascending`, `Descending`, `Mean difference`,
         `Pvalue`) %>% view()


# the effect of change of scale


str_remove("att_01_02", "_[01].")

data_full %>%
  filter(!is.na(exp_pos_change_fct)) %>%
  group_by(exp_pos_change_fct,
           exp_att_fct_2, exp_pos_fct_2, exp_full_fct_2) %>%
  select(all_of(str_c(var_names, "_2"))) %>%
  rename_at(str_c(var_names, "_2"), ~str_remove(., "_2")) %>%
  rename_at(var_names, ~str_remove(., "_")) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T),
                       se = ~sd(., na.rm = T)/sqrt(n()))) %>%
  pivot_longer(
    cols = 5:34,
    values_to = "value",
    names_to = c("var", "stats"),
    names_pattern = "(.+)_(.+)",
  ) %>%
  pivot_wider(
    names_from = stats,
    values_from = value) %>%
  mutate(variable = factor(var, labels = var_labels) %>% fct_rev(),
         lci = mean - (1.96 * se),
         uci = mean + (1.96 * se)) %>%
  ggplot(aes(mean, exp_pos_change_fct,
             alpha = exp_att_fct_2,
             color = exp_pos_fct_2,
             shape = exp_full_fct_2,
             linetype = exp_full_fct_2)) +
  geom_pointrange(aes(xmin = lci, xmax = uci),
                  position = position_dodge(-0.5),
                  line_width = 1, size = 0.15) +
  facet_wrap(~variable, scales = "free", ncol =3) +
  theme_bw() +
  scale_alpha_discrete(range=c(0.4, 0.9)) +
  labs(y = "",
       x = "Mean and 95% confidence intervals",
       alpha = "",
       color = "",
       shape = "",
       linetype = "")



ggsave("./output/fig/mean_dif_change_w2.png",
       dpi = 400, width = 8, height = 6.5)


data_full %>%
  filter(!is.na(exp_pos_change_fct)) %>%
  group_by(exp_pos_change_fct, exp_pos_fct_2,
           exp_att_fct_2, exp_five_fct_2, exp_full_fct_2) %>%
  select(all_of(str_c(var_names, "_2"))) %>%
  summarise_all(.funs =
                  list(mean = ~mean(., na.rm = T))) %>%
  gather(key = key, value = value,
         -exp_pos_change_fct, -exp_pos_fct_2,
         -exp_att_fct_2, -exp_five_fct_2, -exp_full_fct_2) %>%
  group_by(exp_pos_change_fct, exp_pos_fct_2,
           exp_att_fct_2, exp_five_fct_2, exp_full_fct_2) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)

# overall average
data_full %>%
  filter(!is.na(exp_pos_change_fct)) %>%
  group_by(exp_pos_change_fct, exp_pos_fct_2) %>%
  select(all_of(str_c(var_names, "_2"))) %>%
  summarise_all(~mean(., na.rm = T)) %>%
  gather(key = key, value = value,
         -exp_pos_change_fct, -exp_pos_fct_2) %>%
  group_by(exp_pos_change_fct, exp_pos_fct_2) %>%
  summarise(mean = mean(value)) %>%
  arrange(mean)


# t.tests
mean_dif_change <- map(str_c(var_names, "_2"), function(x) {
  t.test(data_full[[x]] ~ data_full$exp_pos_change_fct)
})


cbind(map_dbl(mean_dif_change, function(x) x$p.value) %>% round(2),
      map_df(mean_dif_change, function(x) x$estimate)) %>%
  as_tibble() %>%
  setNames(c("Pvalue", "No change", "Change")) %>%
  mutate(Variable = var_labels,
         'Mean difference' = `No change` - `Change` %>%
           round(3)) %>%
  select(Variable, `No change`, `Change`, `Mean difference`,
         `Pvalue`) %>% view()



## primacy -----------------


# summary statistic wave 1
primacy_summary <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_primacy = mean(primacy, na.rm = T)) %>%
  arrange(mean_primacy)

primacy_summary

primacy_summary %>%
  ggplot(aes(mean_primacy, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average primacy score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/primacy_w1.png", dpi = 300,
       width = 7)


# t-tests
t.test(primacy ~ exp_pos_fct, data = dfw1_2)
t.test(primacy ~ exp_att_fct, data = dfw1_2)
t.test(primacy ~ exp_five_fct, data = dfw1_2)
t.test(primacy ~ exp_full_fct, data = dfw1_2)



# summary statistic wave 2
primacy_summary_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_primacy = mean(primacy, na.rm = T)) %>%
  arrange(mean_primacy)

primacy_summary_w2

primacy_summary_w2 %>%
  ggplot(aes(mean_primacy, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average primacy score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/primacy_w2.png", dpi = 300,
       width = 7)


# t-tests
t.test(primacy ~ exp_pos_fct, data = dfw2_2)
t.test(primacy ~ exp_att_fct, data = dfw2_2)
t.test(primacy ~ exp_five_fct, data = dfw2_2)
t.test(primacy ~ exp_full_fct, data = dfw2_2)





## acquiescence ------------------------------

# summary statistics wave 1
acq_summary <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_acq = mean(acq, na.rm = T)) %>%
  arrange(mean_acq)

acq_summary

acq_summary %>%
  ggplot(aes(mean_acq, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average acquiescence score (% of times chosing positive category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/acq_w1.png", dpi = 300, width = 7)


# t-tests
t.test(acq ~ exp_pos_fct, data = dfw1_2)
t.test(acq ~ exp_att_fct, data = dfw1_2)
t.test(acq ~ exp_five_fct, data = dfw1_2)
t.test(acq ~ exp_full_fct, data = dfw1_2)


# summary statistics wave 2
acq_summary_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_acq = mean(acq, na.rm = T)) %>%
  arrange(mean_acq)

acq_summary_w2

acq_summary_w2 %>%
  ggplot(aes(mean_acq, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average acquiescence score (% of times chosing positive category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/acq_w2.png", dpi = 300, width = 7)


# t-tests
t.test(acq ~ exp_pos_fct, data = dfw2_2)
t.test(acq ~ exp_att_fct, data = dfw2_2)
t.test(acq ~ exp_five_fct, data = dfw2_2)
t.test(acq ~ exp_full_fct, data = dfw2_2)






## straight-lining --------------------------------

# summary statistic w1
same_summary <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_same = mean(same_any, na.rm = T) * 100) %>%
  arrange(mean_same)

same_summary

same_summary %>%
  ggplot(aes(mean_same, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Percentage of straightlining",
       y = "",
       color = "Labels")

ggsave("./output/fig/same_w1.png", dpi = 300, width = 7)


# chi-square
chisq.test(dfw1_2$same_any, dfw1_2$exp_pos_fct)
chisq.test(dfw1_2$same_any, dfw1_2$exp_att_fct)
chisq.test(dfw1_2$same_any, dfw1_2$exp_five_fct)
chisq.test(dfw1_2$same_any, dfw1_2$exp_full_fct)




# summary statistic w2
same_summary_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_same = mean(same_any, na.rm = T) * 100) %>%
  arrange(mean_same)

same_summary_w2

same_summary_w2 %>%
  ggplot(aes(mean_same, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Percentage of straightlining",
       y = "",
       color = "Labels")

ggsave("./output/fig/same_w2.png", dpi = 300, width = 7)


# t-tests
chisq.test(dfw2_2$same_any, dfw2_2$exp_pos_fct)
chisq.test(dfw2_2$same_any, dfw2_2$exp_att_fct)
chisq.test(dfw2_2$same_any, dfw2_2$exp_five_fct)
chisq.test(dfw2_2$same_any, dfw2_2$exp_full_fct)





## extreme -----------------


# summary statistic wave 1
extreme_summary <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_extreme = mean(extreme, na.rm = T)) %>%
  arrange(mean_extreme)

extreme_summary

extreme_summary %>%
  ggplot(aes(mean_extreme, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average extreme score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/extreme_w1.png", dpi = 300,
       width = 7)


# t-tests
t.test(extreme ~ exp_pos_fct, data = dfw1_2)
t.test(extreme ~ exp_att_fct, data = dfw1_2)
t.test(extreme ~ exp_five_fct, data = dfw1_2)
t.test(extreme ~ exp_full_fct, data = dfw1_2)



# summary statistic wave 2
extreme_summary_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_extreme = mean(extreme, na.rm = T)) %>%
  arrange(mean_extreme)

extreme_summary_w2

extreme_summary_w2 %>%
  ggplot(aes(mean_extreme, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average extreme score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/extreme_w2.png", dpi = 300,
       width = 7)


# t-tests
t.test(extreme ~ exp_pos_fct, data = dfw2_2)
t.test(extreme ~ exp_att_fct, data = dfw2_2)
t.test(extreme ~ exp_five_fct, data = dfw2_2)
t.test(extreme ~ exp_full_fct, data = dfw2_2)




## middle -----------------


# summary statistic wave 1
middle_summary <- dfw1_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_middle = mean(middle, na.rm = T)) %>%
  arrange(mean_middle)

middle_summary

middle_summary %>%
  ggplot(aes(mean_middle, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average middle score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/middle_w1.png", dpi = 300,
       width = 7)


# t-tests
t.test(middle ~ exp_pos_fct, data = dfw1_2)
t.test(middle ~ exp_att_fct, data = dfw1_2)
t.test(middle ~ exp_five_fct, data = dfw1_2)
t.test(middle ~ exp_full_fct, data = dfw1_2)



# summary statistic wave 2
middle_summary_w2 <- dfw2_2 %>%
  group_by(exp_pos_fct, exp_att_fct, exp_five_fct, exp_full_fct) %>%
  summarise(mean_middle = mean(middle, na.rm = T)) %>%
  arrange(mean_middle)

middle_summary_w2

middle_summary_w2 %>%
  ggplot(aes(mean_middle, exp_pos_fct, color = exp_full_fct)) +
  facet_grid(exp_att_fct~exp_five_fct) +
  geom_point() +
  theme_bw() +
  labs(x = "Average middle score (% of times chosing 1st category)",
       y = "",
       color = "Labels")

ggsave("./output/fig/middle_w2.png", dpi = 300,
       width = 7)


# t-tests
t.test(middle ~ exp_pos_fct, data = dfw2_2)
t.test(middle ~ exp_att_fct, data = dfw2_2)
t.test(middle ~ exp_five_fct, data = dfw2_2)
t.test(middle ~ exp_full_fct, data = dfw2_2)







## equivalence testing ------------------------------

### wave 1 ------------------

# model 1
model1 <- "f1 =~ att_01 + att_02 + att_03 + att_04 + att_05 + att_06"
fit1 <- cfa(model1, data = dfw1_2, missing = "ML")
summary(fit1, fit.measures = TRUE)


fit1_pos <- cfa(model1, data = dfw1_2, missing = "ML",
                group = "exp_pos")
summary(fit1_pos, fit.measures = TRUE)


fit1_att <- cfa(model1, data = dfw1_2, missing = "ML",
                group = "exp_att")
summary(fit1_att, fit.measures = TRUE)

fit1_five <- cfa(model1, data = dfw1_2, missing = "ML",
                 group = "exp_five")
summary(fit1_five, fit.measures = TRUE)

fit1_full <- cfa(model1, data = dfw1_2, missing = "ML",
                 group = "exp_full")
summary(fit1_full, fit.measures = TRUE)



# model 2
model2 <- "f1 =~ att_07 + att_08 + att_09 + att_10 + att_11 + att_12 + att_13 + att_14 + att_15"
fit2 <- cfa(model2, data = dfw1_2, missing = "ML")
summary(fit2, fit.measures = TRUE)



fit2_pos <- cfa(model2, data = dfw1_2, missing = "ML",
                group = "exp_pos")
summary(fit2_pos, fit.measures = TRUE)


fit2_att <- cfa(model2, data = dfw1_2, missing = "ML",
                group = "exp_att")
summary(fit2_att, fit.measures = TRUE)

fit2_five <- cfa(model2, data = dfw1_2, missing = "ML",
                 group = "exp_five")
summary(fit2_five, fit.measures = TRUE)

fit2_full <- cfa(model2, data = dfw1_2, missing = "ML",
                 group = "exp_full")
summary(fit2_full, fit.measures = TRUE)



# model 1
comp1_pos <- measurementInvariance(model = model1,
                                   data = dfw1_2,
                                   group = "exp_pos")
comp1_att <- measurementInvariance(model = model1,
                                   data = dfw1_2,
                                   group = "exp_att")
comp1_five <- measurementInvariance(model = model1,
                                   data = dfw1_2,
                                   group = "exp_five")
comp1_full <- measurementInvariance(model = model1,
                                   data = dfw1_2,
                                   group = "exp_full")


# model 2
comp2_pos <- measurementInvariance(model = model2,
                                   data = dfw1_2,
                                   group = "exp_pos")


# significant differences in the means
comp2_att <- measurementInvariance(model = model2,
                                   data = dfw1_2,
                                   group = "exp_att")
comp2_five <- measurementInvariance(model = model2,
                                   data = dfw1_2,
                                   group = "exp_five")
comp2_full <- measurementInvariance(model = model2,
                                   data = dfw1_2,
                                   group = "exp_full")



### wave 2 ------------------

# model 1
model1 <- "f1 =~ att_01 + att_02 + att_03 + att_04 + att_05 + att_06"
fit1_w2 <- cfa(model1, data = dfw2_2, missing = "ML")
summary(fit1_w2, fit.measures = TRUE)


fit1_pos_w2 <- cfa(model1, data = dfw2_2, missing = "ML",
                group = "exp_pos")
summary(fit1_pos_w2, fit.measures = TRUE)


fit1_att_w2 <- cfa(model1, data = dfw2_2, missing = "ML",
                group = "exp_att")
summary(fit1_att_w2, fit.measures = TRUE)

fit1_five_w2 <- cfa(model1, data = dfw2_2, missing = "ML",
                 group = "exp_five")
summary(fit1_five_w2, fit.measures = TRUE)

fit1_full_w2 <- cfa(model1, data = dfw2_2, missing = "ML",
                 group = "exp_full")
summary(fit1_full_w2, fit.measures = TRUE)



# model 2
model2 <- "f1 =~ att_07 + att_08 + att_09 + att_10 + att_11 + att_12 + att_13 + att_14 + att_15"
fit2_w2 <- cfa(model2, data = dfw2_2, missing = "ML")
summary(fit2_w2, fit.measures = TRUE)



fit2_pos_w2 <- cfa(model2, data = dfw2_2, missing = "ML",
                group = "exp_pos")
summary(fit2_pos_w2, fit.measures = TRUE)


fit2_att_w2 <- cfa(model2, data = dfw2_2, missing = "ML",
                group = "exp_att")
summary(fit2_att_w2, fit.measures = TRUE)

fit2_five_w2 <- cfa(model2, data = dfw2_2, missing = "ML",
                 group = "exp_five")
summary(fit2_five_w2, fit.measures = TRUE)

fit2_full_w2 <- cfa(model2, data = dfw2_2, missing = "ML",
                 group = "exp_full")
summary(fit2_full_w2, fit.measures = TRUE)



# model 1
comp1_pos_w2 <- measurementInvariance(model = model1,
                                   data = dfw2_2, group = "exp_pos")
comp1_att_w2 <- measurementInvariance(model = model1,
                                   data = dfw2_2, group = "exp_att")
comp1_five_w2 <- measurementInvariance(model = model1,
                                   data = dfw2_2, group = "exp_five")
comp1_full_w2 <- measurementInvariance(model = model1,
                                   data = dfw2_2, group = "exp_full")


# model 2
comp2_pos_w2 <- measurementInvariance(model = model2,
                                   data = dfw2_2, group = "exp_pos")


# significant differences in the means
comp2_att_w2 <- measurementInvariance(model = model2,
                                   data = dfw2_2,
                                   group = "exp_att")

comp2_five_w2 <- measurementInvariance(model = model2,
                                   data = dfw2_2, group = "exp_five")
comp2_full_w2 <- measurementInvariance(model = model2,
                                   data = dfw2_2, group = "exp_full")


## test-retest reliability -----------------------

#
fun_cor <- function(data, var){
  data %>%
    select(starts_with(var)) %>%
    cor(use = "pairwise") %>%
    round(3) %>%
    .[1, 2]
}

# baseline reliability
map_dbl(var_names, fun_cor, data = data_full)

# reliability by group

## positive vs. neg. fist

data_full %>%
  group_by(exp_pos_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x))


rel_pos <- data_full %>%
  group_by(exp_pos_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[1:2]

rel_pos_clean <- rel_pos %>% reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 2) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp = rep(c("Ascending", "Descending"),
                   each = 15))

rel_pos_clean %>%
  ggplot(aes(value, var, color = exp)) +
  geom_point(alpha = 0.8) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "Experimental design") +
  theme_bw()

ggsave("./output/fig/reliability.png", dpi = 300,
       width = 6, height = 4)



rel_pos_clean %>%
  group_by(exp) %>%
  summarise(mean(value))



# reliability graph with full scale
rel_pos_clean %>%
  ggplot(aes(value, var, color = exp)) +
  geom_point(alpha = 0.8) +
  scale_x_continuous(limits = c(0, 1)) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "Experimental design") +
  theme_bw()

ggsave("./output/fig/reliability_full_scale.png", dpi = 300,
       width = 6, height = 4)
ggsave("./share/figure3.pdf", dpi = 300, width = 6, height = 4)



## positive vs. neg. fist

data_full %>%
  group_by(exp_pos_1, exp_pos_change) %>%
  count()

rel_change <- data_full %>%
  group_by(exp_pos_1, exp_pos_change) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1, 2, 4, 5)]

rel_change_clean <- rel_change %>%
  reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 4) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp_pos = rep(c("Ascending", "Descending"),
                       each = 30),
         exp_change = rep(c("No change in direction", "Direction change",
                            "No change in direction", "Direction change"),
                       each = 15))

rel_change_clean %>%
  ggplot(aes(value, var, color = exp_pos, shape = exp_change)) +
  geom_point(alpha = 0.8, position = position_dodge(-0.5)) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "",
       shape = "") +
  theme_bw()

ggsave("./output/fig/reliability_change.png", dpi = 300,
       width = 6, height = 4)

rel_change_clean %>%
  group_by(exp_pos, exp_change) %>%
  summarise(mean(value))



# full vs. end points labels

rel_full <- data_full %>%
  group_by(exp_pos_1, exp_full_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1:4)]


rel_full_clean <- rel_full %>%
  reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 4) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp_pos = rep(c("Ascending", "Descending"),
                       each = 30),
         exp_full = rep(c("Endpoint labels", "Fully laelled",
                            "Endpoint labels", "Fully laelled"),
                          each = 15))

rel_full_clean %>%
  ggplot(aes(value, var, color = exp_pos, shape = exp_full)) +
  geom_point(alpha = 0.8, position = position_dodge(-0.5)) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "",
       shape = "") +
  theme_bw()

ggsave("./output/fig/reliability_labels.png", dpi = 300,
       width = 6, height = 4)

rel_full_clean %>%
  group_by(exp_pos, exp_full) %>%
  summarise(mean(value))





# att vs. behaviors


rel_att <- data_full %>%
  group_by(exp_pos_1, exp_att_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1:4)]


rel_att_clean <- rel_att %>%
  reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 4) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp_pos = rep(c("Ascending", "Descending"),
                       each = 30),
         exp_full = rep(c("Frequency", "Agreement",
                          "Frequency", "Agreement"),
                        each = 15))

rel_att_clean %>%
  ggplot(aes(value, var, color = exp_pos, shape = exp_full)) +
  geom_point(alpha = 0.8, position = position_dodge(-0.5)) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "",
       shape = "") +
  theme_bw()

ggsave("./output/fig/reliability_att.png", dpi = 300,
       width = 6, height = 4)

rel_att_clean %>%
  group_by(exp_pos, exp_full) %>%
  summarise(mean(value))



# five vs. seven point

rel_five <- data_full %>%
  group_by(exp_pos_1, exp_five_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1:4)]


rel_five_clean <- rel_five %>%
  reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 4) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp_pos = rep(c("Ascending", "Descending"),
                       each = 30),
         exp_five = rep(c("Seven point", "Five point",
                          "Seven point", "Five point"),
                        each = 15))

rel_five_clean %>%
  ggplot(aes(value, var, color = exp_pos, shape = exp_five)) +
  geom_point(alpha = 0.8, position = position_dodge(-0.5)) +
  labs(x = "Test-retest reliability",
       y = "Variable",
       color = "",
       shape = "") +
  theme_bw()

ggsave("./output/fig/reliability_five.png", dpi = 300,
       width = 6, height = 4)

rel_five_clean %>%
  group_by(exp_pos, exp_five) %>%
  summarise(mean(value))



rel_five <- data_full %>%
  group_by(exp_pos_1, exp_five_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1:4)]





# testing significance in reliability
rel_all <- data_full %>%
  group_by(exp_pos_1, exp_five_1, exp_att_1, exp_full_1) %>%
  group_map(~map_dbl(var_names, fun_cor, data = .x)) %>%
  .[c(1:16)] %>%
  reduce(cbind) %>%
  as_tibble() %>%
  gather(key = key, value = value) %>%
  mutate(var = rep(var_labels, 16) %>%
           factor(labels = var_labels) %>% fct_rev(),
         exp_pos = rep(c("Ascending", "Descending"),
                       each = 120))

t.test(value ~ exp_pos, data = rel_all)







## mtme analysis ----------


# mtme-1 model for the first scale
mtme_model1 <- "
    T1 =~ 1*att_01_p + 1*att_01_n
    T2 =~ 1*att_02_p + 1*att_02_n
    T3 =~ 1*att_03_p + 1*att_03_n
    T4 =~ 1*att_04_p + 1*att_04_n
    T5 =~ 1*att_05_p + 1*att_05_n
    T6 =~ 1*att_06_p + 1*att_06_n

    P =~ 1*att_01_p + 1*att_02_p + 1*att_03_p + 1*att_04_p + 1*att_05_p + 1*att_06_p

    T1 ~~ T2 + T3 + T4 + T5 + T6
    T2 ~~ T3 + T4 + T5 + T6
    T3 ~~ T4 + T5 + T6
    T4 ~~ T5 + T6
    T5 ~~ T6

    T1 ~~ NA*T1
    T2 ~~ NA*T2
    T3 ~~ NA*T3
    T4 ~~ NA*T4
    T5 ~~ NA*T5
    T6 ~~ NA*T6


    P ~~ NA*P

    P ~~ 0*T1
    P ~~ 0*T2
    P ~~ 0*T3
    P ~~ 0*T4
    P ~~ 0*T5
    P ~~ 0*T6

    T1 + T2 + T3 + T4 + T5 + T6 + P ~ NA*0
"

fit_mtme1 <- lavaan(mtme_model1,
                   data = mtme_data,
                   std.lv = TRUE,
                   missing="ml",
                   auto.fix.first = FALSE,
                   auto.var = TRUE,
                   em.h1.iter.max = 10000)

summary(fit_mtme1, standardized = TRUE, fit.measures = T)


# plot results

mtme_fit_est <- lavaan::parameterestimates(fit_mtme1, standardized = T)

mtme_est_qual <- mtme_fit_est %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "1") ~ "Automatic",
    str_detect(rhs, "2") ~ "Rush",
    str_detect(rhs, "3") ~ "Absorbed",
    str_detect(rhs, "4") ~ "Several",
    str_detect(rhs, "5") ~ "Wanders",
    str_detect(rhs, "6") ~ "Distracted"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n") ~ "Ascending"),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme1_qual_scale_topic.csv")






mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/6) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme1_qual_overall.png", dpi = 300)


mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme1_qual_topic.png", dpi = 300)




# plot quality
mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)/2) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme1_qual_topic_scale.png",
       dpi = 300, width = 10)














# mtme-1 model for scale 2
mtme_model2 <- "

    T7 =~ 1*att_07_p + 1*att_07_n
    T8 =~ 1*att_08_p + 1*att_08_n
    T9 =~ 1*att_09_p + 1*att_09_n
    T10 =~ 1*att_10_p + 1*att_10_n
    T11 =~ 1*att_11_p + 1*att_11_n
    T12 =~ 1*att_12_p + 1*att_12_n
    T13 =~ 1*att_13_p + 1*att_13_n
    T14 =~ 1*att_14_p + 1*att_14_n
    T15 =~ 1*att_15_p + 1*att_15_n

    P =~ 1*att_07_p + 1*att_08_p + 1*att_09_p + 1*att_10_p + 1*att_11_p + 1*att_12_p + 1*att_13_p + 1*att_14_p + 1*att_15_p


    T7 ~~ T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15
    T8 ~~ T9 + T10 + T11 + T12 + T13 + T14 + T15
    T9 ~~ T10 + T11 + T12 + T13 + T14 + T15
    T10 ~~ T11 + T12 + T13 + T14 + T15
    T11 ~~ T12 + T13 + T14 + T15
    T12 ~~ T13 + T14 + T15
    T13 ~~ T14 + T15
    T14 ~~ T15

    T7 ~~ NA*T7
    T8 ~~ NA*T8
    T9 ~~ NA*T9
    T10 ~~ NA*T10
    T11 ~~ NA*T11
    T12 ~~ NA*T12
    T13 ~~ NA*T13
    T14 ~~ NA*T14
    T15 ~~ NA*T15

    P ~~ NA*P

    P ~~ 0*T7
    P ~~ 0*T8
    P ~~ 0*T9
    P ~~ 0*T10
    P ~~ 0*T11
    P ~~ 0*T12
    P ~~ 0*T13
    P ~~ 0*T14
    P ~~ 0*T15

    T7 + T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15 + P ~ NA*0
"

fit_mtme2 <- lavaan(mtme_model2,
                   data = mtme_data,
                   std.lv = TRUE,
                   missing="ml",
                   auto.fix.first = FALSE,
                   auto.var = TRUE,
                   em.h1.iter.max = 10000)

summary(fit_mtme2, standardized = TRUE, fit.measures = T)



# plot results

mtme_fit_est <- lavaan::parameterestimates(fit_mtme2, standardized = T)

mtme_est_qual <- mtme_fit_est %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "7") ~ "Complicated",
    str_detect(rhs, "8") ~ "Understanding",
    str_detect(rhs, "9") ~ "Care",
    str_detect(rhs, "10") ~ "Say",
    str_detect(rhs, "11") ~ "Vote",
    str_detect(rhs, "12") ~ "Journalists",
    str_detect(rhs, "13") ~ "Officials",
    str_detect(rhs, "14") ~ "Choice",
    str_detect(rhs, "15") ~ "Rich"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n") ~ "Ascending",),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))

mtme_est_qual %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme2_qual_scale_topic.csv")






mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/9) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme2_qual_overall.png", dpi = 300)


mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme2_qual_topic.png", dpi = 300)




# plot quality
mtme_est_qual %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme2_qual_topic_scale.png",
       dpi = 300, width = 10)








### mtme sensitivity analysis ----------


## check results if we change the refrence category to positive

# mtme-1 model for the first scale
mtme_model1b <- "
    T1 =~ 1*att_01_p + 1*att_01_n
    T2 =~ 1*att_02_p + 1*att_02_n
    T3 =~ 1*att_03_p + 1*att_03_n
    T4 =~ 1*att_04_p + 1*att_04_n
    T5 =~ 1*att_05_p + 1*att_05_n
    T6 =~ 1*att_06_p + 1*att_06_n

    P =~ 1*att_01_n + 1*att_02_n + 1*att_03_n + 1*att_04_n + 1*att_05_n + 1*att_06_n

    T1 ~~ T2 + T3 + T4 + T5 + T6
    T2 ~~ T3 + T4 + T5 + T6
    T3 ~~ T4 + T5 + T6
    T4 ~~ T5 + T6
    T5 ~~ T6

    T1 ~~ NA*T1
    T2 ~~ NA*T2
    T3 ~~ NA*T3
    T4 ~~ NA*T4
    T5 ~~ NA*T5
    T6 ~~ NA*T6


    P ~~ NA*P

    P ~~ 0*T1
    P ~~ 0*T2
    P ~~ 0*T3
    P ~~ 0*T4
    P ~~ 0*T5
    P ~~ 0*T6

    T1 + T2 + T3 + T4 + T5 + T6 + P ~ NA*0
"

fit_mtme1b <- lavaan(mtme_model1b,
                    data = mtme_data,
                    std.lv = TRUE,
                    missing="ml",
                    auto.fix.first = FALSE,
                    auto.var = TRUE,
                    em.h1.iter.max = 10000)

summary(fit_mtme1b, standardized = TRUE, fit.measures = T)


# plot results

mtme_fit_est_b <- lavaan::parameterestimates(fit_mtme1b,
                                             standardized = T)

mtme_est_qual_b <- mtme_fit_est_b %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "1") ~ "Automatic",
    str_detect(rhs, "2") ~ "Rush",
    str_detect(rhs, "3") ~ "Absorbed",
    str_detect(rhs, "4") ~ "Several",
    str_detect(rhs, "5") ~ "Wanders",
    str_detect(rhs, "6") ~ "Distracted"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n") ~ "Ascending",),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual_b %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual_b %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_b %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual_b %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_b %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme1_qual_scale_topic_b.csv")






mtme_est_qual_b %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/6) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme1_qual_overall_b.png", dpi = 300)


mtme_est_qual_b %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme1_qual_topic_b.png", dpi = 300)




# plot quality
mtme_est_qual_b %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)/2) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme1_qual_topic_scale_b.png",
       dpi = 300, width = 10)














# mtme-1 model for scale 2
mtme_model2b <- "

    T7 =~ 1*att_07_p + 1*att_07_n
    T8 =~ 1*att_08_p + 1*att_08_n
    T9 =~ 1*att_09_p + 1*att_09_n
    T10 =~ 1*att_10_p + 1*att_10_n
    T11 =~ 1*att_11_p + 1*att_11_n
    T12 =~ 1*att_12_p + 1*att_12_n
    T13 =~ 1*att_13_p + 1*att_13_n
    T14 =~ 1*att_14_p + 1*att_14_n
    T15 =~ 1*att_15_p + 1*att_15_n

    P =~ 1*att_07_n + 1*att_08_n + 1*att_09_n + 1*att_10_n + 1*att_11_n + 1*att_12_n + 1*att_13_n + 1*att_14_n + 1*att_15_n


    T7 ~~ T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15
    T8 ~~ T9 + T10 + T11 + T12 + T13 + T14 + T15
    T9 ~~ T10 + T11 + T12 + T13 + T14 + T15
    T10 ~~ T11 + T12 + T13 + T14 + T15
    T11 ~~ T12 + T13 + T14 + T15
    T12 ~~ T13 + T14 + T15
    T13 ~~ T14 + T15
    T14 ~~ T15

    T7 ~~ NA*T7
    T8 ~~ NA*T8
    T9 ~~ NA*T9
    T10 ~~ NA*T10
    T11 ~~ NA*T11
    T12 ~~ NA*T12
    T13 ~~ NA*T13
    T14 ~~ NA*T14
    T15 ~~ NA*T15

    P ~~ NA*P

    P ~~ 0*T7
    P ~~ 0*T8
    P ~~ 0*T9
    P ~~ 0*T10
    P ~~ 0*T11
    P ~~ 0*T12
    P ~~ 0*T13
    P ~~ 0*T14
    P ~~ 0*T15

    T7 + T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15 + P ~ NA*0
"

fit_mtme2b <- lavaan(mtme_model2b,
                    data = mtme_data,
                    std.lv = TRUE,
                    missing="ml",
                    auto.fix.first = FALSE,
                    auto.var = TRUE,
                    em.h1.iter.max = 10000)

summary(fit_mtme2b, standardized = TRUE, fit.measures = T)



# plot results

mtme_fit_est_2b <- lavaan::parameterestimates(fit_mtme2b,
                                              standardized = T)

mtme_est_qual_2b <- mtme_fit_est_2b %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "7") ~ "Complicated",
    str_detect(rhs, "8") ~ "Understanding",
    str_detect(rhs, "9") ~ "Care",
    str_detect(rhs, "10") ~ "Say",
    str_detect(rhs, "11") ~ "Vote",
    str_detect(rhs, "12") ~ "Journalists",
    str_detect(rhs, "13") ~ "Officials",
    str_detect(rhs, "14") ~ "Choice",
    str_detect(rhs, "15") ~ "Rich"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n") ~ "Ascending",),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual_2b %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual_2b %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_2b %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual_2b %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))

mtme_est_qual_2b %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme2_qual_scale_topic_b.csv")






mtme_est_qual_2b %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/9) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme2_qual_overall_b.png", dpi = 300)


mtme_est_qual_2b %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme2_qual_topic_2b.png", dpi = 300)




# plot quality
mtme_est_qual_2b %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme2_qual_topic_scale_b.png",
       dpi = 300, width = 10)


### mtme sensitivity 2 --------------------

# Try with blavaan to deal with estimation issues


future::plan("multicore")
future::plan("multisession")


# mtme-1 model for the first scale
mtme_model1c <- "
    T1 =~ 1*att_01_p + 1*att_01_n
    T2 =~ 1*att_02_p + 1*att_02_n
    T3 =~ 1*att_03_p + 1*att_03_n
    T4 =~ 1*att_04_p + 1*att_04_n
    T5 =~ 1*att_05_p + 1*att_05_n
    T6 =~ 1*att_06_p + 1*att_06_n

    N =~ 1*att_01_n + 1*att_02_n + 1*att_03_n + 1*att_04_n + 1*att_05_n + 1*att_06_n
    P =~ 1*att_01_p + 1*att_02_p + 1*att_03_p + 1*att_04_p + 1*att_05_p + 1*att_06_p


    T1 ~~ T2 + T3 + T4 + T5 + T6
    T2 ~~ T3 + T4 + T5 + T6
    T3 ~~ T4 + T5 + T6
    T4 ~~ T5 + T6
    T5 ~~ T6

    T1 ~~ NA*T1
    T2 ~~ NA*T2
    T3 ~~ NA*T3
    T4 ~~ NA*T4
    T5 ~~ NA*T5
    T6 ~~ NA*T6


    P ~~ NA*P
    P ~~ 0*N
    N ~~ NA*N

    P ~~ 0*T1
    P ~~ 0*T2
    P ~~ 0*T3
    P ~~ 0*T4
    P ~~ 0*T5
    P ~~ 0*T6

    N ~~ 0*T1
    N ~~ 0*T2
    N ~~ 0*T3
    N ~~ 0*T4
    N ~~ 0*T5
    N ~~ 0*T6

    T1 + T2 + T3 + T4 + T5 + T6 + P + N ~ NA*0
"

fit_mtme1c <- lavaan(mtme_model1c,
                     data = mtme_data,
                     std.lv = TRUE,
                     missing="ml",
                     auto.fix.first = FALSE,
                     auto.var = TRUE,
                     em.h1.iter.max = 10000)

summary(fit_mtme1c, standardized = TRUE, fit.measures = T)



# plot results

mtme_fit_est_c <- lavaan::parameterestimates(fit_mtme1c,
                                             standardized = T)

mtme_est_qual_c <- mtme_fit_est_c %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P|N"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "1") ~ "Automatic",
    str_detect(rhs, "2") ~ "Rush",
    str_detect(rhs, "3") ~ "Absorbed",
    str_detect(rhs, "4") ~ "Several",
    str_detect(rhs, "5") ~ "Wanders",
    str_detect(rhs, "6") ~ "Distracted"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n|N") ~ "Ascending"),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P|N") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual_c %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual_c %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_c %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual_c %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_c %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme1_qual_scale_topic_c.csv")






mtme_est_qual_c %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/6) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme1_qual_overall_c.png", dpi = 300)
ggsave("./share/figure4.pdf", dpi = 300)


mtme_est_qual_c %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme1_qual_topic_c.png", dpi = 300)




# plot quality
mtme_est_qual_c %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)/2) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[1:6])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme1_qual_topic_scale_c.png",
       dpi = 300, width = 10)










# mtme model for scale 2
# does not estimate and we try bayesian estimation
mtme_model2c <- "

    T7 =~ 1*att_07_p + 1*att_07_n
    T8 =~ 1*att_08_p + 1*att_08_n
    T9 =~ 1*att_09_p + 1*att_09_n
    T10 =~ 1*att_10_p + 1*att_10_n
    T11 =~ 1*att_11_p + 1*att_11_n
    T12 =~ 1*att_12_p + 1*att_12_n
    T13 =~ 1*att_13_p + 1*att_13_n
    T14 =~ 1*att_14_p + 1*att_14_n
    T15 =~ 1*att_15_p + 1*att_15_n

    N =~ 1*att_07_n + 1*att_08_n + 1*att_09_n + 1*att_10_n + 1*att_11_n + 1*att_12_n + 1*att_13_n + 1*att_14_n + 1*att_15_n

    P =~ 1*att_07_p + 1*att_08_p + 1*att_09_p + 1*att_10_p + 1*att_11_p + 1*att_12_p + 1*att_13_p + 1*att_14_p + 1*att_15_p

    T7 ~~ T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15
    T8 ~~ T9 + T10 + T11 + T12 + T13 + T14 + T15
    T9 ~~ T10 + T11 + T12 + T13 + T14 + T15
    T10 ~~ T11 + T12 + T13 + T14 + T15
    T11 ~~ T12 + T13 + T14 + T15
    T12 ~~ T13 + T14 + T15
    T13 ~~ T14 + T15
    T14 ~~ T15

    T7 ~~ NA*T7
    T8 ~~ NA*T8
    T9 ~~ NA*T9
    T10 ~~ NA*T10
    T11 ~~ NA*T11
    T12 ~~ NA*T12
    T13 ~~ NA*T13
    T14 ~~ NA*T14
    T15 ~~ NA*T15

    P ~~ NA*P
    N ~~ NA*N
    N ~~ 0*P

    P ~~ 0*T7
    P ~~ 0*T8
    P ~~ 0*T9
    P ~~ 0*T10
    P ~~ 0*T11
    P ~~ 0*T12
    P ~~ 0*T13
    P ~~ 0*T14
    P ~~ 0*T15

    N ~~ 0*T7
    N ~~ 0*T8
    N ~~ 0*T9
    N ~~ 0*T10
    N ~~ 0*T11
    N ~~ 0*T12
    N ~~ 0*T13
    N ~~ 0*T14
    N ~~ 0*T15

    T7 + T8 + T9 + T10 + T11 + T12 + T13 + T14 + T15 + P + N ~ NA*0
"

# fit_mtme2c <- blavaan(mtme_model2c,
#                      data = mtme_data,
#                      std.lv = TRUE,
#                      missing="ml",
#                      auto.fix.first = FALSE,
#                      auto.var = TRUE,
#                      em.h1.iter.max = 100000)

# summary(fit_mtme2c, standardized = TRUE, fit.measures = T)



#save(fit_mtme2c, file = "./output/fit_mtme2c.RData")
load("./output/fit_mtme2c.RData")


# plot results

mtme_fit_est_2c <- lavaan::parameterestimates(fit_mtme2c,
                                              standardized = T)

mtme_est_qual_2c <- mtme_fit_est_2c %>%
  filter(op == "=~" | op == "~~") %>%
  filter(!(op == "~~" & str_detect(lhs, "T|M|P|N"))) %>%
  mutate(Trait = case_when(
    str_detect(rhs, "7") ~ "Complicated",
    str_detect(rhs, "8") ~ "Understanding",
    str_detect(rhs, "9") ~ "Care",
    str_detect(rhs, "10") ~ "Say",
    str_detect(rhs, "11") ~ "Vote",
    str_detect(rhs, "12") ~ "Journalists",
    str_detect(rhs, "13") ~ "Officials",
    str_detect(rhs, "14") ~ "Choice",
    str_detect(rhs, "15") ~ "Rich"
  ),
  group = case_when(
    str_detect(rhs, "p|P") ~ "Descending",
    str_detect(rhs, "n|N") ~ "Ascending",),
  source = case_when(str_detect(lhs, "T") ~ "Trait",
                     str_detect(lhs, "P|N") ~ "Direction",
                     TRUE ~ "Random error"),
  qual = ifelse(source != "Random error", std.all^2, std.all)) %>%
  select(-op, -est, -std.lv, -std.nox)



# summarise quality


mtme_est_qual_2c %>%
  group_by(source) %>%
  summarise(qual = mean(qual))

mtme_est_qual_2c %>%
  group_by(source, group) %>%
  summarise(qual = mean(qual))


mtme_est_qual_2c %>%
  group_by(source, Trait) %>%
  summarise(qual = mean(qual)) %>%
  filter(source == "Trait") %>%
  arrange(qual)


mtme_est_qual_2c %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual))

mtme_est_qual_2c %>%
  group_by(Trait, source, group) %>%
  summarise(qual = mean(qual)) %>%
  write_csv("./output/tab/mtme2_qual_scale_topic_c.csv")






mtme_est_qual_2c %>%
  mutate(source = fct_relevel(source, "Trait")) %>%
  group_by(group, source) %>%
  summarise(group_qual = sum(qual)/9) %>%
  ggplot(aes(group, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Measurement type",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))

ggsave("./output/fig/mtme2_qual_overall_c.png", dpi = 300)
ggsave("./share/figure5.pdf", dpi = 300)


mtme_est_qual_2c %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait = fct_rev(Trait)) %>%
  group_by(Trait, source) %>%
  summarise(group_qual = sum(qual)/2) %>%
  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(Trait_fct, group_qual, fill = source)) +
  geom_bar(stat = "identity") +
  #  facet_wrap(~Trait, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Topic",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))
ggsave("./output/fig/mtme2_qual_topic_2c.png", dpi = 300)




# plot quality
mtme_est_qual_2c %>%
  mutate(source = fct_relevel(source, "Trait"),
         Trait= fct_rev(Trait)) %>%
  group_by(Trait, source, group) %>%
  summarise(qual = sum(qual)) %>%

  mutate(Trait_fct = as.factor(Trait),
         Trait_fct = fct_relevel(Trait_fct,
                                 var_labels[7:15])) %>%
  ggplot(aes(group, qual, fill = source)) +
  geom_bar(stat = "identity") +
  facet_wrap(~Trait_fct, nrow = 1) +
  viridis::scale_fill_viridis(discrete = T) +
  labs(y = "Proportion variance",
       x = "Response scale",
       fill = "Source") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = -45,
                                   vjust = 0.5,
                                   hjust = 0.1))


ggsave("./output/fig/mtme2_qual_topic_scale_c.png",
       dpi = 300, width = 10)


