***********************************************************************************************************************************************
***** Stata DO-File for replication of empirical analyses in article "Asking for panel consent in web surveys: Choice, opt-in, or opt-out?" 
***** by Oliver Lipps, Anke Tresch, and Lukas Lauener published in Survey Research Methods
***********************************************************************************************************************************************

global OneDrive "C:\Users\olipps\OneDrive - Université de Lausanne\SRM Submission" // path needs to be adapted
*****

*********************************
*** Data preparation: base file
*********************************

use id W1_f10100 W1_f21310 W2_3_voted W1_f15200 W4_finished W1_mobile W3_f12800? W3_intstart W3_intend using "${OneDrive}\1184_Selects2019_Panel_Data_v5.0.dta", clear // file on SWISSUbase
recode _all (min/-1=.) (98=.) // -99=no answer, 98=don't know

merge 1:1 id using "${OneDrive}\Panel_consent_variables", nogen // Panel consent and survey evaluation variables

rename 	(W1_f21310 W1_f10100 W2_3_voted W1_f15200 W4_finished W1_mobile W3_f30020_ja_1 W3_f30020_ja_2 W3_f30020_ja_3 W3_f30020_ja_4 W3_f30020_nein_1 W3_f30020_nein_2 W3_f30020_nein_3 W3_f30020_nein_4 W3_intstart W3_intend) ///
		(educat politint1 particip1 leftright1 finished4 mobile1 interesting_j3 lengthok_j3 understandable_j3 noproblems_j3 interesting_n3 lengthok_n3 understandable_n3 noproblems_n3 intstart3 intend3)
recode *n3 *j3 (-99=.) // -99=no answer
		
egen trust=rowmean(W3_f12800?) // mean trust in different institutions
foreach var in interesting lengthok understandable noproblems {
 egen `var'3=rowmax(`var'_?3) // ?=j:consenters, ?=n:non-consenters (see variable label)
 replace `var'3=5-`var'3 // higher values mean more positive evaluation
}

recode finished4 (.=0 "attrition or uncompleted (few)"), gen(fini4)
recode politint1 (1=3)(3=1)(4=0) // higher values mean higher polit interest
lab def politint1 0 "Not interested" 1 "Rather not interested" 2 "Rather interested" 3 "Very interested"
lab val politint1 politint1
recode mobile (2=0) // 0=not on mobile
recode educat (14=.) // other

gen byte invited=W3_Consent1==4 | W3_Consent2_4==1 | W3_Consent_2bis_1==1 | W3_Consent3_4==-99 // invited follow-up wave
rename (W3_Consent1 W3_Consent2_4 W3_Consent3_4 W3_Consent_2bis_1) (choice optin optout optin_fu /*follow-up question*/)
egen byte missconsent=rowmiss(choice optin optout optin_fu) // not everybody was asked consent
gen byte noconsent=missconsent==4 if intstart3<. // =1 = were not asked for consent
recode choice (4=1 yes) (5=0 no) (-99=-1 DK), gen(choice_) lab(choice_)
recode optin (-99=0 no) (1=1 ok), gen(optin_complete_) lab(optin_complete_)
replace optin_complete_=1 if optin_complete_==0 & optin_fu==1 // there was a follow-up if no consent in opt-in
recode optout (-99=1 ok) (1=0 no), gen(optout_) lab(optout_)
gen byte opt=choice<.
replace opt=2 if optin_complete_<.
replace opt=3 if optout<.
egen byte agree=rowmax(choice_ optin_complete_ optout_)
recode interesting* lengthok* understandable* noproblems* (-99=.)
recode agree -1=0  // yes/no=DK set to 0

sort intend3 // identify the quota outs (footnote 4)
gen asc=sum(opt==0)
gen byte quotaout=asc>0 & asc<442 // high quotaout, were not asked choice question

gen byte part1=1 // participation in part 1
gen byte part3=noconsent<. // participation in part 3
*download "missings" if not already installed: ssc inst missings
missings tab educat leftright1 politint1 particip1 trust interesting3 lengthok3 understandable3 if agree<. // missings 

*download "ice" if not already installed: ssc install ice
ice educat politint1 particip1 leftright1 mobile1 intstart3 interesting3 lengthok3 understandable3 noproblems3 trust if agree<., clear replace cmd(politint1 particip1 interesting3 lengthok3 understandable3 noproblems3:reg) match(educat politint1 particip1 leftright1 mobile1 intstart3 interesting3 lengthok3 understandable3 noproblems3 trust) seed(1234) // chain imputation of political/survey variables: only those in the analytical sample
replace trust=round(trust,1)
keep if _mj==1

*download "elabel" if not already installed: ssc install elabel 
merge 1:1 PersonId using "${OneDrive}\register" // variables from sampling frame (register)
replace part1=0 if _merge==2 // observations only from register
recode age (18/24=1 "18-24")(25/34=2 "25-34")(35/44=3 "35-44")(45/54=4 "45-54")(55/64=5 "55-64")(65/74=6 "65-74")(75/max=7 "75+"), gen(agecat)
gen byte roman=language>1 if language<.
recode hhsize (4/max=4)
encode marit, gen(marital)
do "${OneDrive}\labels" // variable and value label socio-demographic variables
recode part? (.=0)

drop _mj _mi optin_comp optout_ marit choice optin optout optin_fu choice_ intend finished4 noproblems3 _merge W* interesting_?3 lengthok_?3 understandable_?3 noproblems_?3

save "${OneDrive}\base", replace


***************
*** Analysis
***************

* Table A1
use agecat hhsize marital male roman bornabroad phone using "${OneDrive}\base", clear
gen byte group=1
append using "${OneDrive}\base", keep(agecat hhsize marital male roman bornabroad phone part1) nolab
replace group=2 if group==. & part1==1
append using "${OneDrive}\base", keep(agecat hhsize marital male roman bornabroad phone part3) nolab
replace group=3 if group==. & part3==1
dtable i.agecat i.hhsize i.marital i.male i.roman i.bornabroad i.phone if group<., factor(agecat hhsize marital male roman bornabroad phone, statistics(fvfreq fvperc)) nformat(%9.0g nstats) nformat(%9.1f fstats) by(group, notot)


use "${OneDrive}\base", clear

* Comparisons part 1 and part 3 respondents with full sample
global socdemo i.agecat i.hhsize i.marital i.male i.roman i.bornabroad i.phone
forv j=1(2)3 {
 qui logit part`j' $socdemo // check representativeness socdemo among part1 / part3 wrt full sample
 cap drop part_ 
 predict part_
 qui su part_
 disp "Participation part `j': rindic=" 1-2*r(sd) // r-indicator
}

* Comparison part 1 and part 3 respondents regarding political interest, left-right self-positionning, and education
foreach var in politint1 leftright1 educat { // check if part 3 participants have different polit. attitudes/edu than part 1 participants
 reg `var' part3 if invited<., nohead // 1% level for all variables!
}

keep if agree<. // only the analytical sample

*** Panel consent

* Table 1 (N, mean, se)
tabstat agree, by(opt) s(n mean sem) 

* Are differences between the designs statistically significant?
ttest agree if opt==1 | opt==2, by(opt) // 11% sig. level
ttest agree if opt==1 | opt==3, by(opt) // 1% sig. level
ttest agree if opt==2 | opt==3, by(opt) // 1% sig. level

* Predicted effects on panel consent 
global socdemo 2.opt#quotaout 3.opt#quotaout i.agecat i.hhsize i.marital i.male i.roman i.bornabroad i.phone
global polit educat leftright1 politint1 particip1 trust
global survey interesting3 lengthok3 understandable3 

forv j=1/3 {
 qui logit agree $socdemo if opt==`j' // Table 1, r-indicator socdemo
 cap drop part_ 
 qui predict part_ if opt==`j'
 qui su part_
 disp "N="r(N) ", rindic`j'=" 1-2*r(sd)
}

forv j=1/3 {
 qui logit agree $polit $survey if opt==`j' // Table 1, r-indicator polit/survey
 cap drop part_ 
 qui predict part_ if opt==`j'
 qui su part_ 
 disp "N="r(N) ", rindic`j'=" 1-2*r(sd)
}

* Figure 1
logit agree c.educat##b1.opt
margins opt, dydx(educat)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Education level", size(medium)) ylab(0(.005).03) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r, replace

logit agree c.leftright1##b1.opt
margins opt, dydx(leftright)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Political left-right position", size(medium)) ylab(-.04(.01)0) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r0, replace

logit agree c.politint1##b1.opt
margins opt, dydx(politint1)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Political interest", size(medium)) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r1, replace

logit agree c.interesting3##b1.opt
margins opt, dydx(interesting3)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Survey is interesting", size(medium)) leg(off) ylab(0(.05).25) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r2, replace

logit agree c.particip1##b1.opt
margins opt, dydx(particip1)
marginsplot, leg(row(1)) xtitle("") title("Participated in 2019 elections", size(medium)) ytitle("") ylab(0(.05).25) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r3, replace

logit agree c.lengthok3##b1.opt
margins opt, dydx(lengthok3)
marginsplot, leg(row(1)) xtitle("") title("Length of survey is adequate", size(medium)) ytitle("") leg(off) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r4, replace

logit agree c.trust##b1.opt
margins opt, dydx(trust)
marginsplot, leg(row(1)) xtitle("") title("Trust in institutions", size(medium)) ytitle("") leg(off) ylab(0(.02).06) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r5, replace

logit agree c.understandable3##b1.opt
margins opt, dydx(understandable3)
marginsplot, leg(row(1)) xtitle("") /*xlab(1 `" "Fully or rather disagree" " or neither nor" "' 2 `" "Rather" "agree" "'3 `" "Fully" "agree" "',labsize(small))*/ title("Survey is easy to understand", size(medium)) ytitle("") leg(off) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r6, replace

*download "grc1leg2" if not already installed: ssc install grc1leg2
grc1leg2 r.gph r0.gph r1.gph r5.gph r3.gph r2.gph r4.gph r6.gph, col(4) title("Predicted effects on panel consent", size(medium)) loff
gr save "${OneDrive}\Docs\figure_1", replace
gr use "${OneDrive}\Docs\figure_1.gph"
gr export "${OneDrive}\Docs\figure_1.svg", replace


*** Actual participation in follow-up survey

* Table 2 (N, mean, se)
tabstat fini4, by(opt) s(n mean sem)

* Are differences between the designs statistically significant?
ttest fini4 if inlist(opt,1,2), by(opt) // 50% sig. level
ttest fini4 if inlist(opt,1,3), by(opt) // 1% sig. level
ttest fini4 if inlist(opt,3,2), by(opt) // 1% sig. level

***
forv j=1/3 {
 qui logit fini4 $socdemo if opt==`j' //  Table 2, r-indicator socdemo
 cap drop part_ 
 predict part_ if opt==`j'
 qui su part_
 disp "N="r(N) ", rindic`j'=" 1-2*r(sd)
}

***
forv j=1/3 {
 qui logit fini4 $polit $survey if opt==`j' // Table 2, r-indicator polit/survey
 cap drop part_ 
 predict part_ if opt==`j'
 qui su part_
 disp "N="r(N) ", rindic`j'=" 1-2*r(sd)
}

* Figure 2
logit fini4 c.educat##b1.opt
margins opt, dydx(educat)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Education level", size(medium)) ylab(0(.005).03) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r, replace

logit fini4 c.leftright1##b1.opt
margins opt, dydx(leftright)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Political left-right position", size(medium)) ylab(-.04(.01)0) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r0, replace

logit fini4 c.politint1##b1.opt
margins opt, dydx(politint1) saving(r1, replace)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Political interest", size(medium)) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r1, replace

logit fini4 c.interesting3##b1.opt
margins opt, dydx(interesting3)
marginsplot, leg(row(1)) xtitle("") ytitle("") title("Survey is interesting", size(medium)) leg(off) ylab(0(.05).25) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r2, replace

logit fini4 c.particip1##b1.opt
margins opt, dydx(particip1)
marginsplot, leg(row(1)) xtitle("") title("Participated in 2019 elections", size(medium)) ytitle("") ylab(0(.05).25) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r3, replace

logit fini4 c.lengthok3##b1.opt
margins opt, dydx(lengthok3)
marginsplot, leg(row(1)) xtitle("") title("Length of survey is adequate", size(medium)) ytitle("") leg(off) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r4, replace

logit fini4 c.trust##b1.opt
margins opt, dydx(trust)
marginsplot, leg(row(1)) xtitle("") title("Trust in institutions", size(medium)) ytitle("") leg(off) ylab(0(.02).06) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r5, replace

logit fini4 c.understandable3##b1.opt
margins opt, dydx(understandable3)
marginsplot, leg(row(1)) xtitle("") title("Survey is easy to understand", size(medium)) ytitle("") leg(off) ylab(0(.05).2) plotop(msy(circle) mcol(gs0) lcol(gs0) c(none)) ciopt(lcol(gs0)) yline(0)
gr save r6, replace

grc1leg2 r.gph r0.gph r1.gph r5.gph r3.gph r2.gph r4.gph r6.gph, col(4) title("Predicted effects on participation in follow-up wave", size(medium)) loff
gr save "${OneDrive}\Docs\figure_2.gph", replace
gr use "${OneDrive}\Docs\figure_2.gph"
gr export "${OneDrive}\Docs\figure_2.svg", replace
foreach g in r r0 r1 r2 r3 r4 r5 r6 { // delete intermediate dattafiles and graphs
 erase `g'.gph
}