

# Load data, R-indicators for each round
# R.int.obs <-  readRDS(here::here("R.int.obs.Rdata"))
# glimpse(R.int.obs)

# make wave0
R.int.obs <- R.int.obs %>% mutate(round0 = as.numeric(round) - 1)

# Null model
m0 <- lmerTest::lmer(data = R.int.obs, R ~ 1 + (1 | cntry.full))
summary(m0)


# plot the predicted change

R.int.obs$m0fit <- predict(m0)

R.int.obs %>%
  ggplot(aes(round, R, group = typesamp2, color = typesamp2)) +
  stat_summary(fun = mean, geom = "line", lwd = 1.5) +
  theme_bw() +
  ylim(0.75, 1) +
  labs(x = "Round", y = "R")


# Unconditional change model


ucc <- lmerTest::lmer(data = R.int.obs, R ~ 1 + round0 +
                        (1 + round0 | cntry.full))
summary(ucc)
R.int.obs$uccfit <- predict(ucc)


# Examine the relationship between response rates and R - Conditional change model
cc <- lmerTest::lmer(data = R.int.obs, R ~ 1 + round0 + typesamp2 + 
                       poly((propMean), 2) +
                       (1 + round0 | cntry.full))

summary(cc)
plot(cc)
lattice::qqmath(cc, id=0.05) 

R.int.obs$ccfit <- predict(cc)


sjPlot::tab_model(m0, ucc, cc,
                  dv.labels = c("Empty model", "Unconditional change model", "Conditional change model"),
                  pred.labels = c("Intercept", "Round", "Type of sample: Individual person", "Response Rate", "Response Rate²"), 
                  digits = 3, digits.re = 3,
                  show.r2 = F, show.loglik = T,
                  file = here("Tables", "MLM.doc"))



# Appendix figures Random effects


m0.re.plot <- sjPlot::plot_model(m0, 
                                  type = "re",
                                  vline.color = "black", 
                                  dot.size = 1.5,
                                  show.values = T, 
                                  sort.est = "(Intercept)",
                                  value.offset = .5,
                                  title = ""
)  + 
  ylim(-.25, .2) + 
  theme_minimal() +
  theme(axis.text = element_text(size = 16))

m0.re.plot$data$facet <- factor(m0.re.plot$data$facet)
levels(m0.re.plot$data$facet)
levels(m0.re.plot$data$facet) <- c("Country (Intercept)")
m0.re.plot


ggsave("Figures/random_effects_m0.svg",
       height = 12.5,
       width = 9) 




ucc.re.plot <- sjPlot::plot_model(ucc, 
                                  type = "re",
                                  vline.color = "black", 
                                  dot.size = 1.5,
                                  show.values = T, 
                                  sort.est = "(Intercept)",
                                  value.offset = .5,
                                  title = ""
)  + 
  ylim(-.25, .2) + 
  theme_minimal() +
  theme(axis.text = element_text(size = 16))

ucc.re.plot$data$facet <- factor(ucc.re.plot$data$facet)
levels(ucc.re.plot$data$facet)
levels(ucc.re.plot$data$facet) <- c("Country (Intercept)", "Round")
ucc.re.plot


ggsave("Figures/random_effects_ucc.svg",
       height = 12.5,
       width = 9) 

cc.re.plot <- sjPlot::plot_model(cc, 
                                 type = "re",
                                 vline.color = "black", 
                                 dot.size = 1.5,
                                 sort.est = "(Intercept)",
                                 show.values = T, 
                                 value.offset = .5,
                                 title = "")  + 
  ylim(-.25, .2) + 
  theme_minimal() +
  theme(axis.text = element_text(size = 16))

cc.re.plot$data$facet <- factor(cc.re.plot$data$facet)
levels(cc.re.plot$data$facet)
levels(cc.re.plot$data$facet) <- c("Country (Intercept)", "Round")
cc.re.plot
ggsave("Figures/random_effects_cc.svg",
       height = 12.5,
       width = 9) 





# Appendix Figure 3. Predicted R-indicator, unconditional change model, wide red line denotes grand mean.

R.int.obs %>% 
  filter(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia")) |> 
  select(cntry.full, round_fct, uccfit)

R.int.obs %>% 
  mutate(signi = if_else(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia"), T, F)) %>% 
  ggplot(aes(round, uccfit, group = cntry.full, color = cntry.full, alpha = signi)) + # what variables to plot?
  stat_smooth(method = lm,
              geom = "line",
              alpha = 0.4) + # add a transparent line for each country 
  annotate(geom = "label", x = 4, y = 0.81, label = "Lithuania") +
    annotate(geom = "label", x = 3, y = 0.815, label = "Russia") +
  annotate(geom = "label", x = 2, y = 0.975, label = "Czechia") +
  annotate(geom = "label", x = 1, y = 0.94, label = "Portugal") +
  geom_smooth(data = . %>%  filter(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia")),
              method = lm) +
  stat_smooth( # add average line
    aes(group = 1),
    fun = mean,
    geom = "smooth",
    method = "lm",
    se = F,
    size = 2.5,
    color = "Red"
  ) +
  theme_minimal() + 
  labs(y = "R-indicator", 
       x = "Round", 
       color = "Country") +
  scale_x_continuous(n.breaks = 8, 
                     minor_breaks = seq(1, 9, 1))+
  scale_y_continuous(n.breaks = 8)  +
  theme(legend.position = "none",
        text = element_text(size = 16)) +
  guides(alpha = "none") +
  scale_alpha_manual(values = c(0.2, 1))  

ggsave("Figures/ucc_fit.svg",
       height = 6.6,
       width = 10)


# Figure 5. Predicted R-indicator, conditional change model, wide red line denotes grand mean.


R.int.obs %>% 
  filter(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia", 
                           "Poland", "Ireland", "Ukraine")) |> 
  select(cntry.full, round_fct, ccfit)

R.int.obs %>% 
  mutate(signi = if_else(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia", 
                                           "Poland", "Ireland", "Ukraine"), T, F)) %>% 
  ggplot(aes(round, ccfit, group = cntry.full, color = cntry.full, alpha = signi)) + # what variables to plot?
  stat_smooth(method = lm,
              geom = "line",
              alpha = 0.4) + # add a transparent line for each country 
  annotate(geom = "label", x = 4, y = 0.8, label = "Lithuania") +
  annotate(geom = "label", x = 3, y = 0.81, label = "Russia") +
  annotate(geom = "label", x = 2, y = 0.975, label = "Czechia") +
  annotate(geom = "label", x = 2, y = 0.945, label = "Ukraine") +
  annotate(geom = "label", x = 1, y = 0.945, label = "Ireland") +
  annotate(geom = "label", x = 1, y = 0.85, label = "Poland") +
  annotate(geom = "label", x = 1, y = 0.935, label = "Portugal") +
  geom_smooth(data = . %>%  filter(cntry.full %in% c("Czechia", "Portugal", "Lithuania", "Russia", 
                                                     "Poland", "Ireland", "Ukraine")),
              method = lm, 
              se = F) +
  stat_smooth( # add average line
    aes(group = 1),
    fun = mean,
    geom = "smooth",
    method = "lm",
    se = F,
    size = 2.5,
    color = "Red"
  ) +
  theme_minimal() + 
  labs(y = "R-indicator", 
       x = "Round", 
       color = "Country") +
  scale_x_continuous(n.breaks = 8, 
                     minor_breaks = seq(1, 9, 1))+
  scale_y_continuous(n.breaks = 8)  +
  theme(legend.position = "none",
        text = element_text(size = 16)) +
  ylim(0.78, 1) +
  guides(alpha = "none") +
  scale_alpha_manual(values = c(0.2, 1))    



ggsave("Figures/cc_fit.svg", 
       height = 6.6,
       width = 10)

# Calculate ICC with confidence intervals

calc.icc <- function(y) {
  performance::icc(y)[[1]]
}

calc.icc(m0)
boot.icc <- bootMer(m0, calc.icc, nsim=1000)
quantile(boot.icc$t, c(0.025, 0.975), na.rm = T)

calc.icc(ucc)
boot.icc <- bootMer(ucc, calc.icc, nsim=1000)
quantile(boot.icc$t, c(0.025, 0.975), na.rm = T)

calc.icc(cc)
boot.icc <- bootMer(cc, calc.icc, nsim=1000)
quantile(boot.icc$t, c(0.025, 0.975), na.rm = T)

int <- R.int.obs %>% 
  select(cntry.round, ccfit)



