################################################################################
# Code to process European census data and calculate Duncan Dissimilarity Index
#
# Variable: Age and gender
#
#
# Author: Jan-Lucas Schanze
# Date:   07.10.2022
# Paper: (SRM submission #7830)
################################################################################

### Preparing the data preparation and analysis

# 1. Store raw data (csv files) in a folder     -> directory will be defined as input below
# 2. Create folder for data outputs and graphs  -> directory will be defined as output below

### OVERVIEW OF VARIABLES

# gender:	        0 (male)
#                 1 (female)

# age:            5-year brackets from 0-5 until older than 100 years

### Source for classification of education: https://ec.europa.eu/eurostat/statistics-explained/index.php/International_Standard_Classification_of_Education_(ISCED)#Correspondence_between_ISCED_2011_and_ISCED_1997


######## Installing packages (required once)

#install.packages("foreign")
#install.packages("haven")
#install.packages("Rcmdr")
#install.packages("readr")
#install.packages("lattice")
#install.packages("ggplot2")
#install.packages("tidyverse")
#install.packages("beanplot")
#install.packages("DAAG")
#install.packages("visreg")
#install.packages("HSAUR")
#install.packages("lme4")
#install.packages("readxl")
#install.packages("dplyr")
#install.packages("reshape2")
#install.packages(sjPlot)
#install.packages(mlmRev)
#install.packages("ggmosaic")
#install.packages("gridExtra")

###### Loading packages
library(foreign)
library(haven)
library(Rcmdr)
library(readr)
library(lattice)
library(ggplot2)
library(tidyverse)
library(beanplot)
library(DAAG)
library(visreg)
library(HSAUR)
library(lme4)
library(readxl)
library(dplyr)
library(reshape2)
library(ggmosaic)
library(gridExtra)

# Setting paths for analysis
getwd()

main.path <- "C:\\Users\\schanzjs\\Desktop\\source files\\data"   # change accordingly
save.path <- "C:\\Users\\schanzjs\\Desktop\\source files\\output" # change accordingly 

setwd(main.path)

# Loading dataset

quant <- read_xlsx("censushub_HC39_age_gender.xlsx", sheet = "Data") # Loading dataset with labels
#quant[,1:12] # Have a look at columns 1 through 11

## Dropping countries not needed 
quant <- subset(quant, quant$cntry!="Switzerland") # Dropping CH because of missing information

## Creating a frame giving the totals per country
countries=factor(c(quant$cntry))
gender=factor(c(quant$gender))

# Recoding missing values for all variables
for(i in 1:ncol(quant)){
  quant[quant[,i]=="NA",i] <-  NA
}

# DEFINE as numeric values
quant$ageupto5 <- as.numeric(as.character(quant$ageupto5))
quant$age59 <- as.numeric(as.character(quant$age59))
quant$age1014 <- as.numeric(as.character(quant$age1014))
quant$age1519 <- as.numeric(as.character(quant$age1519))
quant$age2024 <- as.numeric(as.character(quant$age2024))
quant$age2529 <- as.numeric(as.character(quant$age2529))
quant$age3034 <- as.numeric(as.character(quant$age3034))
quant$age3539 <- as.numeric(as.character(quant$age3539))
quant$age4044 <- as.numeric(as.character(quant$age4044))
quant$age4549 <- as.numeric(as.character(quant$age4549))
quant$age5054 <- as.numeric(as.character(quant$age5054))
quant$age5559 <- as.numeric(as.character(quant$age5559))
quant$age6064 <- as.numeric(as.character(quant$age6064))
quant$age6569 <- as.numeric(as.character(quant$age6569))
quant$age7074 <- as.numeric(as.character(quant$age7074))
quant$age7579 <- as.numeric(as.character(quant$age7579))
quant$age8084 <- as.numeric(as.character(quant$age8084))
quant$age8589 <- as.numeric(as.character(quant$age8589))
quant$age9094 <- as.numeric(as.character(quant$age9094))
quant$age9599 <- as.numeric(as.character(quant$age9599))
quant$age100plus <- as.numeric(as.character(quant$age100plus))

### Calculating an AGGREGATE

# Aggregating all countries
quant_eu <- quant 

## checking whether list of countries is correct
prop.table(table(quant_eu$cntry, quant_eu$gender), 2)

quant_eu_men <- subset(quant_eu, quant_eu$gender==0)
quant_eu_women <- subset(quant_eu, quant_eu$gender==1)

### Total number of citizens per age cohort (men)
total_eu_men <- aggregate(cbind(quant_eu_men$ageupto5, 
                            quant_eu_men$age59, 
                            quant_eu_men$age1014,
                            quant_eu_men$age1519,
                            quant_eu_men$age2024, 
                            quant_eu_men$age2529,
                            quant_eu_men$age3034,
                            quant_eu_men$age3539,
                            quant_eu_men$age4044,
                            quant_eu_men$age4549,
                            quant_eu_men$age5054,
                            quant_eu_men$age5559,
                            quant_eu_men$age6064,
                            quant_eu_men$age6569,
                            quant_eu_men$age7074,
                            quant_eu_men$age7579,
                            quant_eu_men$age8084,
                            quant_eu_men$age8589,
                            quant_eu_men$age9094,
                            quant_eu_men$age9599,                              
                            quant_eu_men$age100plus) ~ quant_eu_men$housing, data = quant_eu_men, FUN = function(x) c(gama = sum(x)))

total_eu_men$gender <- c("0")
colnames(total_eu_men)[1] <- "housing" 

### Total number of citizens per age cohort (female)
total_eu_women <- aggregate(cbind(quant_eu_women$ageupto5, 
                                quant_eu_women$age59, 
                                quant_eu_women$age1014,
                                quant_eu_women$age1519,
                                quant_eu_women$age2024, 
                                quant_eu_women$age2529,
                                quant_eu_women$age3034,
                                quant_eu_women$age3539,
                                quant_eu_women$age4044,
                                quant_eu_women$age4549,
                                quant_eu_women$age5054,
                                quant_eu_women$age5559,
                                quant_eu_women$age6064,
                                quant_eu_women$age6569,
                                quant_eu_women$age7074,
                                quant_eu_women$age7579,
                                quant_eu_women$age8084,
                                quant_eu_women$age8589,
                                quant_eu_women$age9094,
                                quant_eu_women$age9599,                              
                                quant_eu_women$age100plus) ~ quant_eu_women$housing, data = quant_eu_women, FUN = function(x) c(gama = sum(x)))

total_eu_women$gender <- c("1")
colnames(total_eu_women)[1] <- "housing" 

## Appending EU-27 data for both genders
total_eu <- rbind(total_eu_women, total_eu_men)

# Generating a country variable
total_eu$cntry <- c("Europe")

colnames(total_eu)[2] <- "ageupto5" 
colnames(total_eu)[3] <- "age59" 
colnames(total_eu)[4] <- "age1014"
colnames(total_eu)[5] <- "age1519"
colnames(total_eu)[6] <- "age2024" 
colnames(total_eu)[7] <- "age2529"
colnames(total_eu)[8] <- "age3034"
colnames(total_eu)[9] <- "age3539"
colnames(total_eu)[10] <- "age4044"
colnames(total_eu)[11] <- "age4549"
colnames(total_eu)[12] <- "age5054"
colnames(total_eu)[13] <- "age5559"
colnames(total_eu)[14] <- "age6064"
colnames(total_eu)[15] <- "age6569"
colnames(total_eu)[16] <- "age7074"
colnames(total_eu)[17] <- "age7579"
colnames(total_eu)[18] <- "age8084"
colnames(total_eu)[19] <- "age8589"
colnames(total_eu)[20] <- "age9094"
colnames(total_eu)[21] <- "age9599"                              
colnames(total_eu)[22] <- "age100plus"

## Appending aggregate data
quant <- rbind(quant, total_eu)

# Removing data frames not required anylonger
rm("quant_eu", "quant_men_eu", "quant_women_eu", "total_eu", "total_eu_men", "total_eu_women")

# Separating the 2 genders 
quant_men <- subset(quant, quant$gender==0)
quant_women <- subset(quant, quant$gender==1)

### Total number of citizens per age cohort
totals <- aggregate(cbind(quant$ageupto5, 
                          quant$age59, 
                          quant$age1014,
                          quant$age1519,
                          quant$age2024, 
                          quant$age2529,
                          quant$age3034,
                          quant$age3539,
                          quant$age4044,
                          quant$age4549,
                          quant$age5054,
                          quant$age5559,
                          quant$age6064,
                          quant$age6569,
                          quant$age7074,
                          quant$age7579,
                          quant$age8084,
                          quant$age8589,
                          quant$age9094,
                          quant$age9599,                              
                          quant$age100plus) ~ quant$cntry, data = quant, FUN = function(x) c(gama = sum(x)))

### Calculating the overall share of institutionalized populations within countries

totals$totpop <-  totals$V1 + totals$V2 + totals$V3 + totals$V4 + totals$V5 + totals$V6 + 
                  totals$V7 + totals$V8 + totals$V9 + totals$V10 + totals$V11 + totals$V12 + 
                  totals$V13 + totals$V14 + totals$V15 + totals$V16 + totals$V17 + totals$V18 + 
                  totals$V19 + totals$V20 + totals$V21

# only 1 variable contains NA for the institutionalized populations -> recode to 0 to avoid listwise deletion
quant2 <- quant
quant2$age100plus[is.na(quant2$age100plus)] <- 0

quant2$all_ages <- quant2$ageupto5 + quant2$age59 + quant2$age1014 + quant2$age1519 + quant2$age2024 +
                  quant2$age2529 + quant2$age3034 + quant2$age3539 + quant2$age4044 + quant2$age4549 +
                  quant2$age5054 + quant2$age5559 + quant2$age6064 + quant2$age6569 + quant2$age7074 +
                  quant2$age7579 + quant2$age8084 + quant2$age8589 + quant2$age9094 + quant2$age9599 + quant2$age100plus

# removing housing types and columns not required anymore
inst_all_ages <- subset(quant2, quant2$housing==2, select=c(cntry, gender, all_ages))

inst_all_ages_male <- subset(inst_all_ages, inst_all_ages$gender==0)
colnames(inst_all_ages_male)[3] <- "all_ages_male"

inst_all_ages_female <- subset(inst_all_ages, inst_all_ages$gender==1)   
colnames(inst_all_ages_female)[3] <- "all_ages_female"

# adding the numbers per country for the 2 genders
inst_all_ages <- aggregate(cbind(inst_all_ages$all_ages) ~ inst_all_ages$cntry, data = inst_all_ages, FUN = function(x) c(gama = sum(x)))
colnames(inst_all_ages)[1] <- "cntry"

# removing variables not required anymore
myvars <- c("quant$cntry", "totpop")
totalpop <- totals[myvars]
colnames(totalpop)[1] <- "cntry"

# generating a dataframe with information from both data frames
rel_size <- merge(totalpop, inst_all_ages, by = "cntry")
colnames(rel_size)[3] <- "inst_all_ages"

rel_size <- merge(rel_size, inst_all_ages_male, by = "cntry")   # merging totals for male population
rel_size <- merge(rel_size, inst_all_ages_female, by = "cntry") # merging totals for female population

# only keeping relevant variables
myvars <- c("cntry", "totpop", "inst_all_ages", "all_ages_male", "all_ages_female")
rel_size <- rel_size[myvars]

# calculating the relative size
rel_size$relative_size        <- rel_size$inst_all_ages/rel_size$totpop # total population
rel_size$relative_size_male   <- rel_size$all_ages_male/rel_size$totpop # male population
rel_size$relative_size_female <- rel_size$all_ages_female/rel_size$totpop # female population

rel_size$relative_weight_male <- round((rel_size$relative_size_male/rel_size$relative_size)*100, digits = 1)

# removing temp dataframes
rm("quant2", "totalpop", "inst_all_ages")


### Total number of MALE citizens per age cohort
totals_men <- aggregate(cbind(quant_men$ageupto5, 
                              quant_men$age59, 
                              quant_men$age1014,
                              quant_men$age1519,
                              quant_men$age2024,
                              quant_men$age2529,
                              quant_men$age3034,
                              quant_men$age3539,
                              quant_men$age4044,
                              quant_men$age4549,
                              quant_men$age5054,
                              quant_men$age5559,
                              quant_men$age6064,
                              quant_men$age6569,
                              quant_men$age7074,
                              quant_men$age7579,
                              quant_men$age8084,
                              quant_men$age8589,
                              quant_men$age9094,
                              quant_men$age9599,                              
                              quant_men$age100plus) ~ quant_men$cntry, data = quant_men, FUN = function(x) c(gama = sum(x)))

### Total number of FEMALE citizens per age cohort
totals_women <- aggregate(cbind(quant_women$ageupto5, 
                                quant_women$age59, 
                                quant_women$age1014,
                                quant_women$age1519,
                                quant_women$age2024,
                                quant_women$age2529,
                                quant_women$age3034,
                                quant_women$age3539,
                                quant_women$age4044,
                                quant_women$age4549,
                                quant_women$age5054,
                                quant_women$age5559,
                                quant_women$age6064,
                                quant_women$age6569,
                                quant_women$age7074,
                                quant_women$age7579,
                                quant_women$age8084,
                                quant_women$age8589,
                                quant_women$age9094,
                                quant_women$age9599,                              
                                quant_women$age100plus) ~ quant_women$cntry, data = quant_women, FUN = function(x) c(gama = sum(x)))

### Merging the total values to the original dataframes
colnames(totals)[1] <- "cntry"
quant <- merge(quant, totals, by = "cntry")

colnames(totals_men)[1] <- "cntry"
quant_men <- merge(quant_men, totals_men, by = "cntry")

colnames(totals_women)[1] <- "cntry"
quant_women <- merge(quant_women, totals_women, by = "cntry")

### Calculating the share for men
quant_men$shareupto5 <- quant_men$ageupto5/quant_men$V1 
quant_men$share59 <- quant_men$age59/quant_men$V2 
quant_men$share1014 <- quant_men$age1014/quant_men$V3
quant_men$share1519 <- quant_men$age1519/quant_men$V4
quant_men$share2024 <- quant_men$age2024/quant_men$V5
quant_men$share2529 <- quant_men$age2529/quant_men$V6
quant_men$share3034 <- quant_men$age3034/quant_men$V7
quant_men$share3539 <- quant_men$age3539/quant_men$V8
quant_men$share4044 <- quant_men$age4044/quant_men$V9
quant_men$share4549 <- quant_men$age4549/quant_men$V10
quant_men$share5054 <- quant_men$age5054/quant_men$V11
quant_men$share5559 <- quant_men$age5559/quant_men$V12
quant_men$share6064 <- quant_men$age6064/quant_men$V13
quant_men$share6569 <- quant_men$age6569/quant_men$V14
quant_men$share7074 <- quant_men$age7074/quant_men$V15
quant_men$share7579 <- quant_men$age7579/quant_men$V16
quant_men$share8084 <- quant_men$age8084/quant_men$V17
quant_men$share8589 <- quant_men$age8589/quant_men$V18
quant_men$share9094 <- quant_men$age9094/quant_men$V19
quant_men$share9599 <- quant_men$age9599/quant_men$V20                              
quant_men$share100plus <- quant_men$age100plus/quant_men$V21

### Calculating the share for women
quant_women$shareupto5 <- quant_women$ageupto5/quant_women$V1 
quant_women$share59 <- quant_women$age59/quant_women$V2 
quant_women$share1014 <- quant_women$age1014/quant_women$V3
quant_women$share1519 <- quant_women$age1519/quant_women$V4
quant_women$share2024 <- quant_women$age2024/quant_women$V5
quant_women$share2529 <- quant_women$age2529/quant_women$V6
quant_women$share3034 <- quant_women$age3034/quant_women$V7
quant_women$share3539 <- quant_women$age3539/quant_women$V8
quant_women$share4044 <- quant_women$age4044/quant_women$V9
quant_women$share4549 <- quant_women$age4549/quant_women$V10
quant_women$share5054 <- quant_women$age5054/quant_women$V11
quant_women$share5559 <- quant_women$age5559/quant_women$V12
quant_women$share6064 <- quant_women$age6064/quant_women$V13
quant_women$share6569 <- quant_women$age6569/quant_women$V14
quant_women$share7074 <- quant_women$age7074/quant_women$V15
quant_women$share7579 <- quant_women$age7579/quant_women$V16
quant_women$share8084 <- quant_women$age8084/quant_women$V17
quant_women$share8589 <- quant_women$age8589/quant_women$V18
quant_women$share9094 <- quant_women$age9094/quant_women$V19
quant_women$share9599 <- quant_women$age9599/quant_women$V20                              
quant_women$share100plus <- quant_women$age100plus/quant_women$V21


### Keeping only the share for institutionalized population 

# For men
quant_men_inst <- subset(quant_men, quant_men$housing==2)

firstcol <- which(colnames(quant_men_inst)=="shareupto5")
lastcol <- which(colnames(quant_men_inst)=="share100plus")

share_men_inst <- subset(quant_men_inst, select = c(1L, firstcol:lastcol))

# For women
quant_women_inst <- subset(quant_women, quant_women$housing==2)

firstcol <- which(colnames(quant_women_inst)=="shareupto5")
lastcol <- which(colnames(quant_women_inst)=="share100plus")

share_women_inst <- subset(quant_women_inst, select = c(1L, firstcol:lastcol))

## create age identifier for visualization
id <- c("cntry", "shareupto5", "share59", "share1014", "share1519", "share2024", "share2529", "share3034", "share3539", "share4044", "share4549", "share5054", "share5559", "share6064", "share6569", "share7074", "share7579", "share8084", "share8589", "share9094", "share9599", "share100plus")
age <- c(0, 0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95, 100)
dfage <- data.frame(id, age)

dfage_t = setNames(data.frame(t(dfage[,-1])), dfage[,1])

share_men_inst <- rbind(share_men_inst, dfage_t)
share_women_inst <- rbind(share_women_inst, dfage_t)

share_men_t = setNames(data.frame(t(share_men_inst[,-1])), share_men_inst[,1])
colname_age <- which(colnames(share_men_t)=="0")
colnames(share_men_t)[colname_age] <- "Age"

share_women_t = setNames(data.frame(t(share_women_inst[,-1])), share_women_inst[,1])
colname_age <- which(colnames(share_women_t)=="0")
colnames(share_women_t)[colname_age] <- "Age"

### Removing frames and objects not needed anymore
rm("dfage", "dfage_t", "age", "colname_age", "id", "firstcol", "lastcol")


### Aggregating age brackets for stacked bars

## For MEN
# removing NA in BG and HR
    quant_men_inst$age100plus[is.na(quant_men_inst$age100plus)] <- 0
    
    quant_men_inst$ageupto15    <- quant_men_inst$ageupto5 + quant_men_inst$age59 + quant_men_inst$age1014 
    quant_men_inst$age1529      <- quant_men_inst$age1519 + quant_men_inst$age2024 + quant_men_inst$age2529 
    quant_men_inst$age3049      <- quant_men_inst$age3034 + quant_men_inst$age3539 + quant_men_inst$age4044 + quant_men_inst$age4549 
    quant_men_inst$age5064      <- quant_men_inst$age5054 + quant_men_inst$age5559 + quant_men_inst$age6064
    quant_men_inst$age6584      <- quant_men_inst$age6569 + quant_men_inst$age7074 + quant_men_inst$age7579 + quant_men_inst$age8084 
    quant_men_inst$age85plus    <- quant_men_inst$age8589 + quant_men_inst$age9094 + quant_men_inst$age9599 + quant_men_inst$age100plus

    quant_men_inst$total_inst <- quant_men_inst$ageupto15 + quant_men_inst$age1529 + quant_men_inst$age3049 + quant_men_inst$age5064 + quant_men_inst$age6584 + quant_men_inst$age85plus
    
    quant_men_inst$total_upto15       <- quant_men_inst$V1 + quant_men_inst$V2 + quant_men_inst$V3 
    quant_men_inst$total_age1529      <- quant_men_inst$V4 + quant_men_inst$V5 + quant_men_inst$V6 
    quant_men_inst$total_age3049      <- quant_men_inst$V7 + quant_men_inst$V8 + quant_men_inst$V9 + quant_men_inst$V10 
    quant_men_inst$total_age5064      <- quant_men_inst$V11 + quant_men_inst$V12 + quant_men_inst$V13 
    quant_men_inst$total_age6584      <- quant_men_inst$V14 + quant_men_inst$V15 + quant_men_inst$V16 + quant_men_inst$V17
    quant_men_inst$total_age85plus    <- quant_men_inst$V18 + quant_men_inst$V19 + quant_men_inst$V20 + quant_men_inst$V21
    
## Distribution (%) WITHIN institutionalized population
quant_men_inst$sharewithinupto15    <- quant_men_inst$ageupto15/quant_men_inst$total_inst
quant_men_inst$sharewithin1529      <- quant_men_inst$age1529/quant_men_inst$total_inst
quant_men_inst$sharewithin3049      <- quant_men_inst$age3049/quant_men_inst$total_inst
quant_men_inst$sharewithin5064      <- quant_men_inst$age5064/quant_men_inst$total_inst
quant_men_inst$sharewithin6584      <- quant_men_inst$age6584/quant_men_inst$total_inst
quant_men_inst$sharewithin85plua    <- quant_men_inst$age85plus/quant_men_inst$total_inst

## Share of institutionalized populations in total population
quant_men_inst$shareupto15    <- quant_men_inst$ageupto15/quant_men_inst$total_upto15
quant_men_inst$share1529      <- quant_men_inst$age1529/quant_men_inst$total_age1529
quant_men_inst$share3049      <- quant_men_inst$age3049/quant_men_inst$total_age3049
quant_men_inst$share5064      <- quant_men_inst$age5064/quant_men_inst$total_age5064
quant_men_inst$share6584      <- quant_men_inst$age6584/quant_men_inst$total_age6584
quant_men_inst$share85plus    <- quant_men_inst$age85plus/quant_men_inst$total_age85plus

## For WOMEN
  quant_women_inst$ageupto15    <- quant_women_inst$ageupto5 + quant_women_inst$age59 + quant_women_inst$age1014 
  quant_women_inst$age1529      <- quant_women_inst$age1519 + quant_women_inst$age2024 + quant_women_inst$age2529 
  quant_women_inst$age3049      <- quant_women_inst$age3034 + quant_women_inst$age3539 + quant_women_inst$age4044 + quant_women_inst$age4549 
  quant_women_inst$age5064      <- quant_women_inst$age5054 + quant_women_inst$age5559 + quant_women_inst$age6064
  quant_women_inst$age6584      <- quant_women_inst$age6569 + quant_women_inst$age7074 + quant_women_inst$age7579 + quant_women_inst$age8084 
  quant_women_inst$age85plus    <- quant_women_inst$age8589 + quant_women_inst$age9094 + quant_women_inst$age9599 + quant_women_inst$age100plus
  
  quant_women_inst$total_inst <- quant_women_inst$ageupto15 + quant_women_inst$age1529 + quant_women_inst$age3049 + quant_women_inst$age5064 + quant_women_inst$age6584 + quant_women_inst$age85plus
  
  quant_women_inst$total_upto15       <- quant_women_inst$V1 + quant_women_inst$V2 + quant_women_inst$V3 
  quant_women_inst$total_age1529      <- quant_women_inst$V4 + quant_women_inst$V5 + quant_women_inst$V6 
  quant_women_inst$total_age3049      <- quant_women_inst$V7 + quant_women_inst$V8 + quant_women_inst$V9 + quant_women_inst$V10 
  quant_women_inst$total_age5064      <- quant_women_inst$V11 + quant_women_inst$V12 + quant_women_inst$V13 
  quant_women_inst$total_age6584      <- quant_women_inst$V14 + quant_women_inst$V15 + quant_women_inst$V16 + quant_women_inst$V17
  quant_women_inst$total_age85plus    <- quant_women_inst$V18 + quant_women_inst$V19 + quant_women_inst$V20 + quant_women_inst$V21

## Distribution (%) WITHIN institutionalized population
quant_women_inst$sharewithinupto15    <- quant_women_inst$ageupto15/quant_women_inst$total_inst
quant_women_inst$sharewithin1529      <- quant_women_inst$age1529/quant_women_inst$total_inst
quant_women_inst$sharewithin3049      <- quant_women_inst$age3049/quant_women_inst$total_inst
quant_women_inst$sharewithin5064      <- quant_women_inst$age5064/quant_women_inst$total_inst
quant_women_inst$sharewithin6584      <- quant_women_inst$age6584/quant_women_inst$total_inst
quant_women_inst$sharewithin85plua    <- quant_women_inst$age85plus/quant_women_inst$total_inst

## Share of institutionalized populations in total population
quant_women_inst$shareupto15    <- quant_women_inst$ageupto15/quant_women_inst$total_upto15
quant_women_inst$share1529      <- quant_women_inst$age1529/quant_women_inst$total_age1529
quant_women_inst$share3049      <- quant_women_inst$age3049/quant_women_inst$total_age3049
quant_women_inst$share5064      <- quant_women_inst$age5064/quant_women_inst$total_age5064
quant_women_inst$share6584      <- quant_women_inst$age6584/quant_women_inst$total_age6584
quant_women_inst$share85plus    <- quant_women_inst$age85plus/quant_women_inst$total_age85plus


## subsetting
agg_brackets_men <- subset(quant_men_inst, select=c(cntry, shareupto15, share1529, share3049, share5064, share6584, share85plus))
agg_brackets_women <- subset(quant_women_inst, select=c(cntry, shareupto15, share1529, share3049, share5064, share6584, share85plus))

# transponing data
agg_brackets_men_t = setNames(data.frame(t(agg_brackets_men[,-1])), agg_brackets_men[,1])
agg_brackets_women_t = setNames(data.frame(t(agg_brackets_women[,-1])), agg_brackets_women[,1])

### Saving data frames for visualization in paper
setwd(save.path)

save(rel_size, file = "rel_size.Rda")

save(share_men_t, file = "share_men_t.Rda")
save(share_men_inst, file = "share_men.Rda")
save(share_women_t, file = "share_women_t.Rda")
save(share_women_inst, file = "share_women.Rda")

save(agg_brackets_men_t, file = "agg_brackets_men_t.Rda")
save(agg_brackets_women_t, file = "agg_brackets_women_t.Rda")

## saving for tables
save(agg_brackets_men, file = "agg_brackets_men.Rda")
save(agg_brackets_women, file = "agg_brackets_women.Rda")

# cleaning up
rm("agg_brackets_men", "agg_brackets_women")


### VISUALIZATION
setwd(save.path)

relsize_dummy <- rel_size %>% 
  select(-totpop, -inst_all_ages, -all_ages_male, -all_ages_female, -relative_weight_male) %>%
  gather(var, value, -cntry, -relative_size) %>% 
  ggplot(., aes(y=reorder(cntry,relative_size), x=value, fill=var)) + 
  geom_col() +
  scale_fill_manual(values = c("gray20","gray67"), name = "Sex", labels = c("Male", "Female")) +
  theme(legend.position="bottom", legend.text = element_text(colour="black", size=8))

# extract the legend in the right order of categories (also for other figures)
legend_genders <- get_legend(relsize_dummy)

# display the figure
relsize_genders <- rel_size %>% 
  select(-totpop, -inst_all_ages, -all_ages_male, -all_ages_female, -relative_weight_male) %>%
  gather(var, value, -cntry, -relative_size) %>% 
  #ggplot(., aes(y=reorder(cntry,relative_size), x=value, fill=var)) + 
  ggplot(., aes(y=reorder(cntry,relative_size), x=value)) + 
  geom_col() +
  scale_fill_manual(values = alpha(c("gray67", "gray20"), 8/10), name = "Gender", labels = c("Female", "Male")) +
  scale_y_discrete(name="") +
  scale_x_continuous(name="", limits=c(0, 0.03), labels = scales::percent) +
  theme_light() 

relsize_genders <- relsize_genders + theme(legend.position = "none") # suppressing the legend for this plot

#fig_relsize <- grid.arrange(relsize_genders, 
#                            legend_genders, nrow = 2, ncol = 1,
#                            widths = c(10), heights = c(10, 0.5))

## Plotting the figure
relsize_genders



### -------------------------------------
#       OVERVIEW FOR ALL COUNTRIES
### -------------------------------------


## which countries are available
prop.table(table(rel_size$cntry,rel_size$totpop), 2)
      
#------------------------------------------------#
#               EUROPE
#------------------------------------------------#
      
rel_freq_cntry <-  ggplot(share_men_t, aes(y=Europe, x=Age)) +
                    geom_area(color='gray20', alpha = 2/10) +
                    geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
                    scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
                    scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
                    ggtitle("Europe") + theme(plot.title = element_text(hjust = 0.5)) + 
                    theme_light()

out <- grid.arrange(rel_freq_cntry, 
                   legend_genders, nrow = 2, ncol = 1,
                   widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Europe.jpg", out)



#------------------------------------------------#
#               AUSTRIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Austria, x=Age)) +
                    geom_area(color='gray20', alpha = 2/10) +
                    geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
                    scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
                    scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
                    ggtitle("Austria") + theme(plot.title = element_text(hjust = 0.5)) + 
                    theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Austria.jpg", out)



#------------------------------------------------#
#               BELGIUM
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Belgium, x=Age)) +
                    geom_area(color='gray20', alpha = 2/10) +
                    geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
                    scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
                    scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
                    ggtitle("Belgium") + theme(plot.title = element_text(hjust = 0.5)) + 
                    theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Belgium.jpg", out)



#------------------------------------------------#
#               BULGARIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Bulgaria, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Bulgaria") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Bulgaria.jpg", out)



#------------------------------------------------#
#               CROATIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Croatia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Croatia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Croatia.jpg", out)



#------------------------------------------------#
#               CYPRUS
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Cyprus, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Cyprus") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Cyprus.jpg", out)


#------------------------------------------------#
#               CZECHIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Czechia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Czechia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Czechia.jpg", out)


#------------------------------------------------#
#               DENMARK
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Denmark, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Denmark") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Denmark.jpg", out)


#------------------------------------------------#
#               ESTONIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Estonia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Estonia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Estonia.jpg", out)


#------------------------------------------------#
#               FINLAND
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Finland, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Finland") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Finland.jpg", out)


#------------------------------------------------#
#               FRANCE
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=France, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("France") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_France.jpg", out)


#------------------------------------------------#
#               GERMANY
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Germany, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Germany") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Germany.jpg", out)


#------------------------------------------------#
#               GREECE
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Greece, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Greece") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Greece.jpg", out)



#------------------------------------------------#
#               HUNGARY
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Hungary, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Hungary") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Hungary.jpg", out)


#------------------------------------------------#
#               ICELAND
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Iceland, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Iceland") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Iceland.jpg", out)


#------------------------------------------------#
#               IRELAND
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Ireland, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Ireland") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Ireland.jpg", out)



#------------------------------------------------#
#               ITALY
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Italy, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Italy") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Italy.jpg", out)


#------------------------------------------------#
#               LATVIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Latvia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Latvia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Latvia.jpg", out)


#------------------------------------------------#
#               LITHUANIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Lithuania, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Lithuania") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Lithuania.jpg", out)



#------------------------------------------------#
#               LUXEMBOURG
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Luxembourg, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Luxembourg") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Luxembourg.jpg", out)


#------------------------------------------------#
#               MALTA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Malta, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Malta") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Malta.jpg", out)


#------------------------------------------------#
#               NETHERLANDS
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Netherlands, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Netherlands") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Netherlands.jpg", out)



#------------------------------------------------#
#               NORWAY
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Norway, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Norway") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Norway.jpg", out)


#------------------------------------------------#
#               POLAND
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Poland, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Poland") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Poland.jpg", out)


#------------------------------------------------#
#               PORTUGAL
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Portugal, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Portugal") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Portugal.jpg", out)


#------------------------------------------------#
#               ROMANIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Romania, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Romania") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Romania.jpg", out)


#------------------------------------------------#
#               SLOVAKIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Slovakia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Slovakia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Slovakia.jpg", out)


#------------------------------------------------#
#               SLOVENIA
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Slovenia, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Slovenia") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Slovenia.jpg", out)


#------------------------------------------------#
#               SPAIN
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Spain, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Spain") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Spain.jpg", out)



#------------------------------------------------#
#               SWEDEN
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=Sweden, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("Sweden") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_Sweden.jpg", out)


#------------------------------------------------#
#               UNITED KINGDOM
#------------------------------------------------#

rel_freq_cntry <-  ggplot(share_men_t, aes(y=UK, x=Age)) +
  geom_area(color='gray20', alpha = 2/10) +
  geom_area(data = share_women_t, color='gray67', alpha = 2/10) +
  scale_y_continuous(name="% of respective group", limits=c(0, 0.6), labels = scales::percent) +
  scale_x_continuous(name = NULL, breaks = seq(0, 100, by = 10)) +
  ggtitle("United Kingdom") + theme(plot.title = element_text(hjust = 0.5)) + 
  theme_light()

out <- grid.arrange(rel_freq_cntry, 
                    legend_genders, nrow = 2, ncol = 1,
                    widths = c(10), heights = c(5, 0.5))

ggsave("age_gender_UK.jpg", out)