################################################################################
### functions used in other code files 

################################################################################
account_pop_move<-function (aggr_data_votes_elect_1, aggr_data_votes_elect_2, method = c("hawkes")) 
{
  library(checkmate)
  method <- match.arg(method)
  if (method == "hawkes") {
    aggr_data <- account_pop_move_hawkes(aggr_data_votes_elect_1, 
                                         aggr_data_votes_elect_2)
  }
}

################################################################################
account_pop_move_hawkes<-function (votes_elect_1, votes_elect_2) 
{
  row_sum_elect_1 = rowSums(votes_elect_1, na.rm = TRUE)
  row_sum_elect_2 = rowSums(votes_elect_2, na.rm = TRUE)
  shares_elect_1 = votes_elect_1/row_sum_elect_1
  votes_elect_1 = shares_elect_1 * row_sum_elect_2
  votes_elect_1_fract = votes_elect_1%%1
  votes_elect_1_fract_min = votes_elect_1_fract
  votes_elect_1_fract_min[votes_elect_1_fract_min < 0.5] = NA
  votes_elect_1_fract_max = votes_elect_1_fract
  votes_elect_1_fract_max[votes_elect_1_fract_max >= 0.5] = NA
  votes_elect_1 = round(votes_elect_1)
  row_sum_elect_1 = rowSums(votes_elect_1, na.rm = TRUE)
  diff_row_sum <- row_sum_elect_1 - row_sum_elect_2
  diff_not_0 <- diff_row_sum != 0
  diff_not_0 <- seq(1, nrow(votes_elect_1))[diff_not_0]
  for (i in diff_not_0) {
    if (diff_row_sum[i] > 0) {
      col_min_fract = which.min(votes_elect_1_fract_min[i, 
                                                        ])
      votes_elect_1[i, col_min_fract] = votes_elect_1[i, 
                                                      col_min_fract] - diff_row_sum[i]
    }
    else {
      col_max_fract = which.max(votes_elect_1_fract_max[i, 
                                                        ])
      votes_elect_1[i, col_max_fract] = votes_elect_1[i, 
                                                      col_max_fract] - diff_row_sum[i]
    }
  }
  return(list(votes_elect_1 = votes_elect_1, votes_elect_2 = votes_elect_2))
}

################################################################################
aggr_data_besip<-function (data_besip, country = c("all", "eng", "wal", 
                                  "sco"), parties = c("CON", "LAB", "LD", 
                                                      "UKIP", "SNP", "PC", "GREEN"), default_parties = TRUE, 
          election_names = c("10", "15"), method = "hawkes") 
{
  country = match.arg(country)
  aggr_besip <- data_besip[, c(1, 31:54, 59:60, 62:65)]
  names(aggr_besip)[2:31] <- toupper(names(aggr_besip)[2:31])
  aggr_besip <- unique_data(aggr_besip, "district")
  id <- aggr_besip$district
  aggr_data_shares <- aggr_besip[, c(2:7, 14:19, 29, 31)]
  aggr_data_votes <- aggr_besip[, c(8:13, 20:27, 28, 30)]
  names(aggr_data_votes) <- gsub("VOTE", "", names(aggr_data_votes))
  col_names_aggr_data = colnames(aggr_data_votes)
  elect_1_col = grep(election_names[1], col_names_aggr_data)
  elect_2_col = grep(election_names[2], col_names_aggr_data)
  aggr_data_votes_elect_1 = aggr_data_votes[, elect_1_col]
  aggr_data_votes_elect_2 = aggr_data_votes[, elect_2_col]
  aggr_data_votes_elect_1 = calc_abstain(aggr_data_votes_elect_1, 
                                         election_names[1])
  aggr_data_votes_elect_2 = calc_abstain(aggr_data_votes_elect_2, 
                                         election_names[2])
  list_aggr_data <- account_pop_move(aggr_data_votes_elect_1, 
                                     aggr_data_votes_elect_2)
  election_1 = list_aggr_data$votes_elect_1
  election_2 = list_aggr_data$votes_elect_2
  if (!default_parties) {
    election_1 <- select_parties_aggr(election_1, parties, 
                                      election_names[1])
    election_2 <- select_parties_aggr(election_2, parties, 
                                      election_names[2])
    parties <- union(parties, "OTHER")
  }
  aggr_data <- cbind(district = id, election_1, election_2)
  aggr_data[is.na(aggr_data)] <- 0
  sort_part_1 <- paste(parties, election_names[1], sep = "")
  sort_part_2 <- paste(parties, election_names[2], sep = "")
  aggr_data <- aggr_data[, c("district", sort_part_1, 
                             sort_part_2)]
  return(aggr_data)
}

################################################################################
ballot_comp_green<-function (data_besip, method = c("with_greeen", "without_green", 
                                 "green_small")) 
{
  method <- match.arg(method)
  data_besip$GreenVote15[is.na(data_besip$GreenVote15)] = 0
  data_besip$GreenVote10[is.na(data_besip$GreenVote10)] = 0
  if (method == "with_greeen") {
    data_besip = data_besip[(data_besip$GreenVote15 != 0) & 
                              (data_besip$GreenVote10 != 0), ]
  }
  else if (method == "green_small") {
    data_besip = data_besip[(data_besip$GreenVote15 != 0) & 
                              (data_besip$GreenVote10 != 0), ]
    data_besip = data_besip[(data_besip$GreenVote15 < 10000) & 
                              (data_besip$GreenVote10 < 5000), ]
  }
  else if (method == "without_green") {
    data_besip = data_besip[(data_besip$GreenVote15 == 0) & 
                              (data_besip$GreenVote10 == 0), ]
  }
  return(data_besip)
}

################################################################################
ballot_comp_ukip<-function (data_besip, method = c("with_ukip", "without_ukip")) 
{
  method <- match.arg(method)
  data_besip$UKIPVote15[is.na(data_besip$UKIPVote15)] = 0
  data_besip$UKIPVote10[is.na(data_besip$UKIPVote10)] = 0
  if (method == "with_ukip") {
    data_besip = data_besip[(data_besip$UKIPVote15 != 0) & 
                              (data_besip$UKIPVote10 != 0), ]
  }
  else if (method == "without_ukip") {
    data_besip = data_besip[(data_besip$UKIPVote15 == 0) & 
                              (data_besip$UKIPVote10 == 0), ]
  }
  return(data_besip)
}

################################################################################
besip_ipfp<-function (besip_data, aggr_data, indi_data, country = c("all", 
                                                        "eng", "wal", "sco"), election_names) 
{
  country = match.arg(country)
  margins = calc_marg_prob(aggr_data = aggr_data, election_names = election_names)
  trans_matrix_list = transition_matrix_individual_data(indi_data)
  trans_matrix = trans_matrix_list$trans_matrix_abs
  trans_matrix_rel = trans_matrix_list$trans_matrix_rel
  if (country == "eng") {
    trans_matrix = trans_matrix[c(2, 1, 3, 4, 5, 6), c(2, 
                                                       1, 3, 4, 5, 6)]
    trans_matrix_rel = trans_matrix_rel[c(2, 1, 3, 4, 5, 
                                          6), c(2, 1, 3, 4, 5, 6)]
  }
  else if (country == "sco") {
    trans_matrix = trans_matrix[c(1, 2, 3, 5, 4), c(1, 2, 
                                                    3, 5, 4)]
    trans_matrix_rel = trans_matrix_rel[c(1, 2, 3, 5, 4), 
                                        c(1, 2, 3, 5, 4)]
  }
  else if (country == "wal") {
    trans_matrix = trans_matrix[c(1, 2, 3, 5, 4), c(1, 2, 
                                                    3, 5, 4)]
    trans_matrix_rel = trans_matrix_rel[c(1, 2, 3, 5, 4), 
                                        c(1, 2, 3, 5, 4)]
  }
  numb_pers = sum(trans_matrix)
  margin_row = numb_pers * margins$marg_prob_elec_1
  margin_row = margin_row[row.names(trans_matrix)]
  margin_col = numb_pers * margins$marg_prob_elec_2
  margin_col = margin_col[colnames(trans_matrix)]
  trans_matrix_ipfp = ipfhk(trans_matrix, mr = margin_row, 
                            mc = margin_col)
  trans_matrix_ipfp_rel = prop.table(trans_matrix_ipfp, 1)
  trans_matrix_ipfp = round(trans_matrix_ipfp)
  marg_prob_elec_1 = apply(trans_matrix, 1, sum)/numb_pers
  marg_prob_elec_2 = apply(trans_matrix, 2, sum)/numb_pers
  indi_ipfp = list(marg_prob_elec_1 = marg_prob_elec_1, marg_prob_elec_2 = marg_prob_elec_2)
  margins_prob = list(aggr_data = margins, indi_ipfp = indi_ipfp)
  return(list(trans_matrix_ipfp_abs = trans_matrix_ipfp, trans_matrix_ipfp_rel = trans_matrix_ipfp_rel, 
              trans_matrix_indi_abs = trans_matrix, trans_matrix_indi_rel = trans_matrix_rel, 
              margins_prob = margins_prob))
}

################################################################################ 
besip_model<-function (besip_data, country = c("all", "eng", "wal", 
                                  "sco"), parties = c("CON", "LAB", "LD", 
                                                      "UKIP", "SNP", "PC", "GREEN"), election_names = c("10", 
                                                                                                        "15"), individual = FALSE, method, sample = 2000, thinning = 2, 
          prioriPars = NULL, burnin = 1500, tune_vars = FALSE, maxiter = 20, 
          alphaVars = NULL, betaVars = NULL, verbose = 100, get_only_aggr_indi_data = FALSE, 
          weighted = FALSE) 
{
  country = match.arg(country)
  if (country == "all") {
    default_parties <- all(c("CON", "LAB", "LD", 
                             "UKIP", "SNP", "PC", "GREEN") %in% 
                             parties)
  }
  else if (country == "eng") {
    default_parties <- all(c("CON", "LAB", "LD", 
                             "UKIP", "GREEN") %in% parties)
  }
  else if (country == "wal") {
    default_parties <- all(c("CON", "LAB", "LD", 
                             "UKIP", "PC", "GREEN") %in% parties)
  }
  else if (country == "sco") {
    default_parties <- all(c("CON", "LAB", "LD", 
                             "UKIP", "SNP", "GREEN") %in% parties)
  }
  if (!("ABSTAIN" %in% parties)) 
    parties = c(parties, "ABSTAIN")
  aggr_besip <- aggr_data_besip(besip_data, country = country, 
                                default_parties = default_parties, parties = parties)
  if (individual) {
    indi_list <- indi_data_besip(besip_data, default_parties = default_parties, 
                                 parties = parties, weighted = weighted)
    indi_besip <- indi_list$indi_besip
    indi_like_aggr <- indi_list$indi_like_aggr
    indi_list_check <- check_indi_data(aggr_besip, indi_like_aggr, 
                                       indi_besip)
    indi_besip <- indi_list_check$indi_besip
    indi_like_aggr <- indi_list_check$indi_like_aggr
  }
  else {
    indi_besip <- NULL
    indi_like_aggr = NULL
  }
  if (!get_only_aggr_indi_data) {
    mod_besip <- hyb_mult_dirich_mod(aggr_besip, indi_besip, 
                                     parties = parties, default_parties = default_parties, 
                                     election_names = election_names, sample = sample, 
                                     thinning = thinning, prioriPars = prioriPars, burnin = burnin, 
                                     tune_vars = tune_vars, maxiter = maxiter, alphaVars = alphaVars, 
                                     betaVars = betaVars, verbose = verbose)
  }
  else {
    mod_besip <- list(model = NULL, alphaVars = NULL, betaVars = NULL)
  }
  return(list(model = mod_besip$model, aggr = aggr_besip, indi = indi_besip, 
              alphaVars = mod_besip$alphaVars, betaVars = mod_besip$betaVars, 
              indi_like_aggr = indi_like_aggr, sample = sample, thinning = thinning, 
              burnin = burnin))
}

################################################################################
box_plot_margin<-function (data, y = c("votes", "shares"), title = "Boxplot", 
          parties = NULL) 
{
  library(ggplot2)
  y = match.arg(y)
  numb_parties = (ncol(data) - 1)/2
  data$votes_district = apply(data[, 2:(numb_parties + 1)], 
                              1, sum)
  data = reshape_aggr_data_box_plot(data)
  if (y == "shares") {
    data$value = data$value/data$votes_district
  }
  if (is.null(parties)) {
    parties = unique(data$party)
  }
  data$party = factor(data$party, levels = parties)
  ylab_box = y
  ylim_max <- max(0.75, data$value)
  ggplot(data, aes(party, value)) + geom_boxplot(aes(fill = factor(election_name))) + 
    ggtitle(title) + theme(axis.title.x = element_blank(), 
                           axis.title.y = element_text(size = 25), axis.text.y = element_text(size = 23), 
                           axis.text.x = element_text(size = 23), legend.title = element_text(size = 23), 
                           legend.text = element_text(size = 23), title = element_text(size = 26), 
                           plot.title = element_text(hjust = 0.5)) + ylab("Anteil") + 
    ylim(0, ylim_max) + scale_fill_discrete(name = "Wahl", 
                                            breaks = c("10", "15"), labels = c("2010", 
                                                                               "2015"))
}

################################################################################ 
calc_abstain<-function (aggr_data, election_name) 
{
  col_name_abstain = paste("ABSTAIN", election_name, 
                           sep = "")
  col_all_voters = grep("ELECTORATE", colnames(aggr_data))
  all_voters = aggr_data[, col_all_voters]
  aggr_data = aggr_data[, -col_all_voters]
  sum_row_sub = rowSums(aggr_data, na.rm = TRUE)
  aggr_data[, col_name_abstain] = all_voters - sum_row_sub
  return(aggr_data)
}

################################################################################ 
calc_abstain_2<-function (aggr_data, election_name) 
{
  col_name_abstain = paste("ABSTAIN", election_name, 
                           sep = "")
  col_all_voters = grep("ELECTORATE", colnames(aggr_data))
  all_voters = aggr_data[, col_all_voters]
  aggr_data = aggr_data[, -col_all_voters]
  invalid = aggr_data[, "invalid"]
  sum_row_sub = rowSums(aggr_data, na.rm = TRUE)
  aggr_data[, col_name_abstain] = all_voters - sum_row_sub + 
    invalid
  return(aggr_data)
}

################################################################################ 
calc_ad<-function (abs_tab_elect_1, abs_tab_elect_2) 
{
  if (!identical(dim(abs_tab_elect_1), dim(abs_tab_elect_2))) 
    stop("Transition tables should have same dimensions.")
  prob_tab_1 = prop.table(abs_tab_elect_1)
  prob_tab_2 = prop.table(abs_tab_elect_2)
  tab_dist = abs(prob_tab_1 - prob_tab_2)
  ad = sum(tab_dist)
  return(ad)
}

################################################################################ 
calc_LAB10<-function (aggr_data_shares, aggr_data_votes) 
{
  aggr_data_votes$LABVOTE10 = aggr_data_shares$LAB10 * aggr_data_votes$ELECTORATE.2010/100
  aggr_data_votes$LABVOTE10 = round(aggr_data_votes$LABVOTE10)
  return(aggr_data_votes)
}

################################################################################ 
calc_mae<-function (rel_tab_elect_1, rel_tab_elect_2) 
{
  if (!identical(dim(rel_tab_elect_1), dim(rel_tab_elect_1))) 
    stop("Transition tables should have same dimensions.")
  n_cell = prod(dim(rel_tab_elect_1))
  tab_dist = abs(rel_tab_elect_1 - rel_tab_elect_2)
  mae = (1/n_cell) * sum(tab_dist)
  return(mae)
}

################################################################################ 
calc_marg_prob<-function (aggr_data, election_names) 
{
  col_elec_1 = grep(election_names[1], names(aggr_data))
  col_elec_2 = grep(election_names[2], names(aggr_data))
  aggr_data_elec_1 = aggr_data[, col_elec_1]
  aggr_data_elec_2 = aggr_data[, col_elec_2]
  sum_elec_1 = apply(aggr_data_elec_1, 2, sum)
  sum_elec_2 = apply(aggr_data_elec_2, 2, sum)
  marg_prob_elec_1 = sum_elec_1/sum(sum_elec_1)
  marg_prob_elec_2 = sum_elec_2/sum(sum_elec_2)
  return(list(marg_prob_elec_1 = marg_prob_elec_1, marg_prob_elec_2 = marg_prob_elec_2))
}

################################################################################ 
check_aggr_data_shares<-function (data_shares, perc = c(TRUE, FALSE), transform = c(TRUE, 
                                                            FALSE)) 
{
  ifelse(perc, complete <- 100, complete <- 1)
  shares_row_sum <- apply(data_shares, 1, function(x) sum(x, 
                                                          na.rm = TRUE))
  not_complete_rows <- shares_row_sum != complete
  ifelse(sum(not_complete_rows) == 0, all_rows_complete <- TRUE, 
         all_rows_complete <- FALSE)
  if (transform & !all_rows_complete) {
    data_shares[not_complete_rows, ] <- data_shares[not_complete_rows, 
                                                    ]/shares_row_sum[not_complete_rows] * 100
  }
  output <- list(all_rows_complete = all_rows_complete, data_shares = data_shares, 
                 not_complete_rows = not_complete_rows)
}

################################################################################ 
check_indi_data<-function (aggr_besip, indi_like_aggr, indi_besip) 
{
  aggr_besip <- aggr_besip[order(aggr_besip$district), ]
  indi_like_aggr <- indi_like_aggr[order(indi_like_aggr$district), 
                                   ]
  diff_aggr_indi <- aggr_besip - indi_like_aggr
  diff_aggr_indi$district <- aggr_besip$district
  numb_neg_cells <- sum(diff_aggr_indi < 0)
  if (numb_neg_cells > 0) {
    indi_list <- delete_indi_data(diff_aggr_indi, indi_besip)
    indi_besip <- indi_list$indi_besip
    indi_like_aggr <- indi_list$indi_like_aggr
  }
  return(list(indi_besip = indi_besip, indi_like_aggr = indi_like_aggr))
}

################################################################################ 
compare_transition_rates<-function (indi_data, group_var) 
{
  numb_parties <- length(levels(indi_data$vote))
  sum_indi_data <- summarize_indi_data(indi_data, group_var, 
                                       weighted = weighted)
  col_group_var <- which(names(sum_indi_data) == group_var)
  uni_group_var <- unique(sum_indi_data[, col_group_var])
  empt_matrix <- matrix(0, numb_parties, numb_parties)
  rownames(empt_matrix) <- levels(indi_data$vote)
  colnames(empt_matrix) <- levels(indi_data$vote)
  tables_group_var <- rep(list(empt_matrix), length(uni_group_var))
  names(tables_group_var) <- uni_group_var
  for (i in 1:length(uni_group_var)) {
    rows_i <- sum_indi_data[, col_group_var] == uni_group_var[i]
    subset_i <- sum_indi_data[rows_i, c("vote_numb", 
                                        "recall_numb", "count")]
    coord_i <- (subset_i$vote_numb - 1) * numb_parties + 
      subset_i$recall_numb
    tables_group_var[[i]][coord_i] <- subset_i$count
    row_sum <- apply(tables_group_var[[i]], 1, sum)
    not_zero <- row_sum != 0
    tables_group_var[[i]][not_zero, ] <- tables_group_var[[i]][not_zero, 
                                                               ]/row_sum[not_zero]
    tables_group_var[[i]] <- round(tables_group_var[[i]] * 
                                     100, 1)
  }
  return(tables_group_var)
}

################################################################################ 
create_comp_tab<-function (models_vect, method = c("ad", "mae")) 
{
  method = match.arg(method)
  n_models = length(models_vect)
  tab = matrix(ncol = n_models, nrow = n_models)
  colnames(tab) = names(models_vect)
  row.names(tab) = names(models_vect)
  for (i in 1:n_models) {
    for (j in 1:n_models) {
      if (method == "ad") {
        tab[i, j] = calc_ad(models_vect[[i]], models_vect[[j]])
      }
      if (method == "mae") {
        tab[i, j] = calc_mae(models_vect[[i]], models_vect[[j]])
      }
    }
  }
  tab = round(100 * tab, 2)
  return(tab)
}

################################################################################ 
data_party_district_long<-function (data) 
{
  numb_parties = (ncol(data) - 1)/2
  data$votes_district = apply(data[, 2:(numb_parties + 1)], 
                              1, sum)
  data = reshape_aggr_data_box_plot(data)
  data$share = data$value/data$votes_district
  data = data[, c("district", "election_name", 
                  "party", "share")]
  return(data)
}

################################################################################ 
data_scatter_plot<-function (data_aggr, data_indi) 
{
  indi_long = data_party_district_long(data_indi)
  aggr_long = data_party_district_long(data_aggr)
  names(aggr_long)[names(aggr_long) == "share"] = "share_aggr"
  names(indi_long)[names(indi_long) == "share"] = "share_indi"
  data = merge(aggr_long, indi_long, by = c("district", 
                                            "election_name", "party"))
  return(data)
}

################################################################################
delete_indi_data<-function (diff_aggr_indi, indi_besip) 
{
  neg_diff_aggr_indi <- diff_aggr_indi < 0
  coord_neg <- which(neg_diff_aggr_indi, arr.ind = TRUE)
  length_coord_neg <- nrow(coord_neg)
  party_names_aggr <- colnames(diff_aggr_indi)
  party_names_indi <- colnames(indi_besip)
  for (i in 1:length_coord_neg) {
    distr_i <- diff_aggr_indi$district[coord_neg[i, 1]]
    party_aggr_i <- party_names_aggr[coord_neg[i, 2]]
    col_indi_i <- grep(party_aggr_i, party_names_indi)
    indi_besip[indi_besip$district == distr_i, col_indi_i] <- 0
  }
  indi_like_aggr <- indi_like_aggr_after_deleting(indi_besip)
  return(list(indi_besip = indi_besip, indi_like_aggr = indi_like_aggr))
}

################################################################################ 
hyb_mult_dirich_mod<-function (aggr_data, indi_data = NULL, parties, default_parties = TRUE, 
          election_names, sample = 2000, burnin = 1500, thinning = 2, 
          tune_vars = FALSE, prioriPars = list(shape = 4, rate = 2), 
          maxiter = 20, alphaVars = NULL, betaVars = NULL, verbose = 100) 
{
  library(eiwild)
  if (!default_parties) {
    beg_parties <- parties
    end_parties <- "OTHER"
  }
  else {
    numb_parties <- length(parties)
    beg_parties <- parties[1:numb_parties - 1]
    end_parties <- parties[numb_parties]
  }
  if (is.null(prioriPars)) 
    prioriPars <- list(shape = 4, rate = 2)
  beg_parties_1 <- paste(beg_parties, election_names[1], ", ", 
                         sep = "")
  end_parties_1 <- paste(end_parties, election_names[1], sep = "")
  part_elect_1 <- paste(c(beg_parties_1, end_parties_1), collapse = "")
  beg_parties_2 <- paste(beg_parties, election_names[2], ", ", 
                         sep = "")
  end_parties_2 <- paste(end_parties, election_names[2], sep = "")
  part_elect_2 <- paste(c(beg_parties_2, end_parties_2), collapse = "")
  form <- as.formula(paste("cbind(", part_elect_2, ") ~ cbind(", 
                           part_elect_1, ")"))
  if (tune_vars) {
    vars <- tuneVars(form = form, aggr = aggr_data, indi = indi_data, 
                     IDCols = c("district", "district"), maxiter = maxiter, 
                     prioriPars = prioriPars)
    alphaVars = vars$alphaVars
    betaVars = vars$betaVars
  }
  hyb_mult_dirich <- indAggEi(form = form, aggr = aggr_data, 
                              indi = indi_data, IDCols = c("district", "district"), 
                              sample = sample, alphaVars = alphaVars, betaVars = betaVars, 
                              thinning = thinning, burnin = burnin, verbose = verbose, 
                              prioriPars = prioriPars)
  return(list(model = hyb_mult_dirich, alphaVars = alphaVars, 
              betaVars = betaVars))
}

################################################################################
indi_data_besip<-function (indi_data, default_parties = TRUE, parties = NULL, 
          election_names = c("10", "15"), weighted = weighted) 
{
  if (!("ABSTAIN" %in% parties)) 
    parties = c(parties, "ABSTAIN")
  indi_data <- indi_data[, c("district", "vote", 
                             "recall", "wt_full_W6")]
  levels_parties <- toupper(gsub("UKID", "UKIP", 
                                 levels(indi_data$vote)))
  levels_parties <- gsub("GRE", "GREEN", levels_parties)
  levels(indi_data$vote) <- levels_parties
  levels(indi_data$recall) <- levels_parties
  if (!default_parties) {
    indi_data$vote <- select_parties_indi(indi_data$vote, 
                                          parties)
    indi_data$recall <- select_parties_indi(indi_data$recall, 
                                            parties)
    parties <- union(parties, "OTHER")
  }
  indi_data$vote <- droplevels(indi_data$vote)
  levels_parties <- levels(indi_data$vote)
  sum_indi_data <- summarize_indi_data(indi_data, group_var = "district", 
                                       weighted = weighted)
  vote_elect_name <- paste(sum_indi_data$vote, election_names[2], 
                           sep = "")
  recall_elect_name <- paste(sum_indi_data$recall, election_names[1], 
                             sep = "")
  sum_indi_data$col_name <- paste(recall_elect_name, vote_elect_name, 
                                  sep = ".")
  numb_distr <- length(unique(indi_data$district))
  numb_part <- length(levels_parties)
  indi_besip <- data.frame(matrix(0, nrow = numb_distr, ncol = (numb_part^2 + 
                                                                  1)))
  part_1 <- paste(levels_parties, election_names[1], sep = "")
  part_2 <- paste(levels_parties, election_names[2], sep = "")
  all_comb_part <- expand.grid(part_1, part_2)
  all_comb_part <- all_comb_part[order(all_comb_part$Var1), 
                                 ]
  all_comb_part <- as.vector(paste(all_comb_part$Var1, all_comb_part$Var2, 
                                   sep = "."))
  names(indi_besip) <- c("district", all_comb_part)
  unique_distr <- unique(sum_indi_data$district)
  indi_besip[, 1] <- unique_distr
  for (i in 1:numb_distr) {
    sub_distr_i <- subset(sum_indi_data, district == unique_distr[i])
    indi_besip[i, sub_distr_i$col_name] <- sub_distr_i$count
  }
  indi_like_aggr <- indi_table_like_aggr(sum_indi_data, parties, 
                                         election_names)
  list(indi_besip = indi_besip, indi_like_aggr = indi_like_aggr)
}

################################################################################ 
indi_like_aggr_after_deleting<-function (indi_data) 
{
  col_distr = which(names(indi_data) == "district")
  col_names = names(indi_data)[-col_distr]
  marg_part_elect_1 = sub("\\..*", "", col_names)
  part_elect_1 = unique(marg_part_elect_1)
  marg_part_elect_2 = sub(".*\\.", "", col_names)
  part_elect_2 = unique(marg_part_elect_2)
  ncol_tab = length(part_elect_1) + length(part_elect_2) + 
    1
  indi_like_aggr = data.frame(matrix(0, ncol = ncol_tab, nrow = nrow(indi_data)))
  names(indi_like_aggr) = c("district", part_elect_1, 
                            part_elect_2)
  indi_like_aggr$district = indi_data$district
  for (i in 1:nrow(indi_data)) {
    votes_i = as.numeric(indi_data[i, -col_distr])
    marg_sum_elect_1 = tapply(votes_i, marg_part_elect_1, 
                              sum)
    indi_like_aggr[i, names(marg_sum_elect_1)] = marg_sum_elect_1
    marg_sum_elect_2 = tapply(votes_i, marg_part_elect_2, 
                              sum)
    indi_like_aggr[i, names(marg_sum_elect_2)] = marg_sum_elect_2
  }
  return(indi_like_aggr)
}

################################################################################ 
indi_table_like_aggr<-function (sum_indi_data, parties, election_names) 
{
  unique_distr <- unique(sum_indi_data$district)
  numb_distr <- length(unique_distr)
  numb_part <- length(parties)
  indi_like_aggr <- data.frame(matrix(0, nrow = numb_distr, 
                                      ncol = (numb_part * 2 + 1)))
  indi_like_aggr[, 1] <- unique_distr
  names_elect_1 <- paste(parties, election_names[1], sep = "")
  names_elect_2 <- paste(parties, election_names[2], sep = "")
  names(indi_like_aggr) <- c("district", names_elect_1, 
                             names_elect_2)
  sum_indi_data[, c("vote")] <- paste(sum_indi_data[, 
                                                    c("vote")], election_names[2], sep = "")
  sum_indi_data[, c("recall")] <- paste(sum_indi_data[, 
                                                      c("recall")], election_names[1], sep = "")
  for (i in 1:numb_distr) {
    sub_distr_i <- subset(sum_indi_data, district == unique_distr[i])
    sum_part_vote <- tapply(sub_distr_i$count, INDEX = sub_distr_i$vote, 
                            function(x) sum(x, na.rm = TRUE))
    part_distr_vote_i <- names(sum_part_vote)
    indi_like_aggr[i, part_distr_vote_i] <- sum_part_vote
    sum_part_recall <- tapply(sub_distr_i$count, INDEX = sub_distr_i$recall, 
                              function(x) sum(x, na.rm = TRUE))
    part_distr_recall_i <- names(sum_part_recall)
    indi_like_aggr[i, part_distr_recall_i] <- sum_part_recall
  }
  return(indi_like_aggr)
}

################################################################################ 
reshape_aggr_data_box_plot<-function (data) 
{
  library(reshape)
  library(stringi)
  data_resh = melt(data, id.var = c("district", "votes_district"))
  data_resh$election_name = stri_sub(data_resh$variable, -2, 
                                     -1)
  data_resh$party = apply(data_resh[, c("election_name", 
                                        "variable")], 1, function(x) {
                                          gsub(x[1], "", x[2])
                                        })
  data_resh = data_resh[, c("district", "votes_district", 
                            "election_name", "party", "value")]
  return(data_resh)
}

################################################################################ 
select_parties_aggr<-function (data_aggr, parties, election_name) 
{
  names(data_aggr) <- gsub(election_name, "", names(data_aggr))
  other_col <- setdiff(names(data_aggr), parties)
  other <- as.data.frame(data_aggr[, other_col])
  colnames(other) <- "ABSTAIN"
  other_sum <- apply(other, 1, function(x) sum(x, na.rm = TRUE))
  data_aggr_select <- data.frame(data_aggr[, parties], OTHER = other_sum)
  names(data_aggr_select) <- paste(names(data_aggr_select), 
                                   election_name, sep = "")
  data_aggr_select
}

################################################################################ 
select_parties_indi<-function (data_indi, parties) 
{
  all_parties <- levels(data_indi)
  other_parties <- setdiff(all_parties, parties)
  data_indi <- as.character(data_indi)
  data_indi[data_indi %in% other_parties] <- "OTHER"
  data_indi <- factor(data_indi, levels = c(parties, "OTHER"))
}

################################################################################ 
summarize_indi_data<-function (indi_data, group_var, weighted = weighted) 
{
  levels_parties <- levels(indi_data$vote)
  col_group_var <- which(names(indi_data) == group_var)
  indi_data$vote_numb <- factor(indi_data$vote, levels = levels_parties, 
                                labels = seq(1, length(levels_parties)))
  indi_data$vote_numb <- as.integer(indi_data$vote_numb)
  indi_data$recall_numb <- factor(indi_data$recall, levels = levels_parties, 
                                  labels = seq(1, length(levels_parties)))
  indi_data$recall_numb <- as.integer(indi_data$recall_numb)
  indi_data$id <- as.factor(paste(indi_data[, col_group_var], 
                                  indi_data$vote_numb, indi_data$recall_numb, sep = "."))
  if (!weighted) {
    indi_data$count <- 1
  }
  else {
    indi_data$count <- indi_data$wt_full_W6
  }
  sum_indi_data <- aggregate(indi_data$count, by = list(indi_data$id), 
                             sum, na.rm = TRUE)
  names(sum_indi_data) <- c("id", "count")
  col_merge <- c("id", group_var, "vote", "vote_numb", 
                 "recall", "recall_numb")
  sum_indi_data <- merge(unique(indi_data[, col_merge]), sum_indi_data, 
                         by = "id")
}

################################################################################ 
transition_matrix_individual_data<-function (indi_data) 
{
  col_sum_indi = colSums(indi_data[-1])
  col_sum_indi = col_sum_indi[col_sum_indi != 0]
  names_indi = names(col_sum_indi)[col_sum_indi != 0]
  elect_1 = sub("\\..*", "", names_indi)
  uni_elect_1 = unique(elect_1)
  elect_2 = sub(".*\\.", "", names_indi)
  party_names = gsub("10", "", uni_elect_1)
  trans_matrix = matrix(0, ncol = length(party_names), nrow = length(party_names))
  row.names(trans_matrix) = party_names
  colnames(trans_matrix) = party_names
  for (i in seq_along(unique(elect_1))) {
    party_i = col_sum_indi[elect_1 == uni_elect_1[i]]
    vote_party_i = sub(".*\\.", "", names(party_i))
    vote_party_i = gsub("15", "", vote_party_i)
    trans_matrix[i, vote_party_i] = party_i
  }
  row.names(trans_matrix) = paste(row.names(trans_matrix), 
                                  "10", sep = "")
  colnames(trans_matrix) = paste(colnames(trans_matrix), "15", 
                                 sep = "")
  trans_matrix_rel = prop.table(trans_matrix, 1)
  return(list(trans_matrix_abs = trans_matrix, trans_matrix_rel = trans_matrix_rel))
}

################################################################################ 
unique_data<-function (data, id) 
{
  library(checkmate)
  assert_data_frame(data)
  assert_character(id, min.len = 1, max.len = 1)
  col_id <- grep(id, names(data))
  unique_distr <- !duplicated(data[, col_id])
  unique_data <- data[unique_distr, ]
}

                                                                                
################################################################################
#functions transition flows                                                                                                                        
calc.sald = function(data.abs){
  calc.sald.func = function(data.calc = data.abs){
    sald = data.calc
    for (i in 1:nrow(sald)){
      for (j in i:ncol(sald)){
        sald[i,j] = data.calc[i,j] - data.calc[j,i]
        sald[j,i] = data.calc[j,i] - data.calc[i,j]
      }
    }
    return(sald)
  }
  parties2   = unlist(strsplit(colnames(data.abs), "_2"))
  parties1   = unlist(strsplit(rownames(data.abs), "_1"))
  both.elecs  = match(parties2, parties1)
  both.elecs2 = match(parties1, parties2)
  sald = data.abs
  if ( sum(is.na(both.elecs)) == 0  & sum(is.na(both.elecs2)) == 0){
    t.tab = data.abs[,both.elecs]
    sald  = calc.sald.func(t.tab)
  } else {
    only.1 = parties1[is.na(match(parties1, parties2))]
    only.2 = parties2[is.na(match(parties2, parties1))]
    t.tab  = data.abs[both.elecs[!is.na(both.elecs)], both.elecs2[!is.na(both.elecs2)]]
    t.tab.sald = calc.sald.func(t.tab)
    sald[!(parties1 %in% only.1),!(parties2 %in% only.2)] = t.tab.sald
  }
  return(sald)
}

################################################################################
logit  = function(x){log(x/(1-x))}

################################################################################
plot.sald.grid = function(data.abs = table.abs, party.sel = "NW", direct = 1,  title.txt = "", round.v = 100, min = 500,
                          colv1 = c("black","red","green3","yellow2","magenta","orange","gray20","gray40"),
                          colv2 = c("black","red","green3","yellow2","magenta","orange","gray20","gray40"),
                          coll1 = c( rep("white",2), rep("black",4), rep("white",2)),
                          coll2 = c( rep("white",2), rep("black",4), rep("white",2)),
                          cex.axis = 0.75, cex.text = 1, min.txt = 0.30){
  parties2   = unlist(strsplit(colnames(data.abs), "_2"))
  parties1   = unlist(strsplit(rownames(data.abs), "_1"))
  col.n    = match(party.sel, parties2)
  row.n    = match(party.sel, parties1)
  calc.mat = calc.sald(data.abs)
  if (is.na(row.n)){
    sald.weg = rep(0,length(parties2))
  } else {
    sald.weg = calc.sald(data.abs)[row.n,]
  }
  if (is.na(col.n)){
    sald.zu = rep(0,length(parties1))
  } else {
    sald.zu  = calc.sald(data.abs)[,col.n]
  }
  sald.weg.r = round.v*(round(sald.weg/round.v,0))
  sald.zu.r  = round.v*(round(sald.zu /round.v,0))
  calc.mat2 = calc.mat
  calc.mat2[abs(calc.mat2) <= min] = 0
  calc.mat3 = round.v*round(calc.mat2/round.v,0)
  calc.mat4 = calc.mat3
  calc.mat4[calc.mat4 < 0] = 0
  calc.mat.weg.sum = rowSums(calc.mat4)
  calc.mat.zu.sum  = colSums(calc.mat4)
  calc.mat.weg.rel = calc.mat.weg.sum/rowSums(data.abs)
  calc.mat.zu.rel  = calc.mat.zu.sum/colSums(data.abs)
  # Gitter f?r plot
  order.y.e1 = order(calc.mat.weg.sum[calc.mat.weg.sum > 0.1], decreasing = TRUE)
  order.y.e2 = order(calc.mat.zu.sum[calc.mat.zu.sum > 0.1]  , decreasing = TRUE)
  grid.y.e1  = prop.table(calc.mat.weg.sum[calc.mat.weg.sum > 0.1])
  grid.y.e2  = prop.table(calc.mat.zu.sum[calc.mat.zu.sum > 0.1])
  grid.x.l   = c(logit((1:999)/1000))
  grid.x.l   = 5.0 + 2.5*grid.x.l/max(grid.x.l)*0.99
  grid.y.l   = (0:998)/999
  x.vec    = c(grid.x.l, grid.x.l[length(grid.x.l):1])
  box.dist   = 0.05
  plot(0,0, col = NA, xaxt = "n", yaxt = "n", xlim = c(0,10), ylim = c(0,10), main = title.txt, xlab = "", ylab = "", bty = "n", xaxs = "i", yaxs = "i")
  axis(1, at = c(0,.5,1,1.5,2,2.5), labels = paste(c(100,80,60,40,20,0),"%", sep = ""), cex.axis = cex.axis, col.axis = "gray", col = "gray")
  axis(1, at = 1.25, labels = "Share loss", line = 2, col.axis = "gray", cex.axis = cex.axis, tick = FALSE)
  axis(1, at = 7.5+c(0,.5,1,1.5,2,2.5), labels = paste(c(100,80,60,40,20,0),"%", sep = "")[6:1], cex.axis = cex.axis, col.axis = "gray", col = "gray")
  axis(1, at = 7.5+1.25, labels = "Share profit", line = 2, col.axis = "gray", cex.axis = cex.axis, tick = FALSE)
  cur.y = 10
  for (i in 1:length(order.y.e1)){
    j = order.y.e1[i]
    cur.y.t = cur.y
    cur.y   = cur.y - (10-2*box.dist*length(order.y.e1))*grid.y.e1[j] - 2*box.dist
    if ( sum(parties2 %in% parties1[calc.mat.weg.sum > 0][j]) > .1){
      cur.x   = 2.5-(2.5-2*box.dist)*calc.mat.weg.rel[calc.mat.weg.sum > 0][j]
      polygon(c(0,cur.x-box.dist,cur.x-box.dist,0),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv1[calc.mat.weg.sum > 0.1][j])
      polygon(c(cur.x+box.dist,2.5,2.5,cur.x+box.dist),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv1[calc.mat.weg.sum > 0.1][j])
    } else {
      polygon(c(0,2.5,2.5,0),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv1[calc.mat.weg.sum > 0.1][j])
    }
    if ( (cur.y.t - cur.y) > min.txt){
      text(1*box.dist, cur.y.t-2*box.dist, parties1[calc.mat.weg.sum > 0.1][j], adj = c(0,1), col = coll1[calc.mat.weg.sum > 0.1][j],
           cex = cex.text)
    }
  }
  cur.y = 10
  cur.e2        = rep(0, sum(calc.mat.zu.sum > 0.1))
  cur.e2.height = rep(0, sum(calc.mat.zu.sum > 0.1))
  for (i in 1:length(order.y.e2)){
    j = order.y.e2[i]
    cur.y.t = cur.y
    cur.y   = cur.y - (10-2*box.dist*length(order.y.e2))*grid.y.e2[j] - 2*box.dist
    cur.e2[j]        = cur.y.t - box.dist
    cur.e2.height[j] = cur.y.t - cur.y - 2*box.dist
    if ( sum(parties1 %in% parties2[calc.mat.zu.sum > 0][j]) > .1){
      cur.x   = 7.5+(2.5-2*box.dist)*calc.mat.zu.rel[calc.mat.zu.sum > 0][j]
      polygon(c(7.5,cur.x-box.dist,cur.x-box.dist,7.5),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv2[calc.mat.zu.sum > 0.1][j])
      polygon(c(cur.x+box.dist,10,10,cur.x+box.dist),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv2[calc.mat.zu.sum > 0.1][j])
    } else {
      polygon(c(7.5,10,10,7.5),c(cur.y + box.dist, cur.y + box.dist, cur.y.t - box.dist, cur.y.t - box.dist), col = colv2[calc.mat.zu.sum > 0.1][j])
    }
    if ( (cur.y.t - cur.y) > min.txt){
      text(10-1*box.dist, cur.y.t-2*box.dist, parties2[calc.mat.zu.sum > 0.1][j], adj = c(1,1), col = coll2[calc.mat.zu.sum > 0.1][j],
           cex = cex.text)
    }
  }
  cur.y  = 10
  cur.e2.back = cur.e2
  for (i in 1:length(order.y.e1)){
    j = order.y.e1[i]
    cur.y.t = cur.y
    cur.y   = cur.y - (10-2*box.dist*length(order.y.e1))*grid.y.e1[j] - 2*box.dist
    cur.line     = (calc.mat4[calc.mat.weg.sum > 0.1,][j,])[calc.mat.zu.sum > 0.1]
    cur.line.rel = prop.table(cur.line)
    cur.box.height  = cur.y.t - cur.y - 2*box.dist
    cur.line.e1     = cur.y.t-box.dist
    cur.col   = rgb(red = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[1,1], green = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[2,1], blue = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[3,1], alpha = 60, maxColorValue = 255)
    cur.col.l = rgb(red = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[1,1], green = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[2,1], blue = col2rgb(colv1[calc.mat.weg.sum > 0.1][j])[3,1], alpha = 120, maxColorValue = 255)
    for (k in 1:length(cur.line)){
      if (cur.line[order.y.e2[k]] > 0.1){
        temp.box.height  = cur.box.height * cur.line.rel[order.y.e2[k]]
        cur.line.e1.top  = cur.line.e1
        cur.line.e1      = cur.line.e1 - temp.box.height
        cur.line.e2.top  = cur.e2[order.y.e2[k]]
        cur.line.e2      = cur.e2[order.y.e2[k]] - cur.e2.height[order.y.e2[k]]*cur.line[order.y.e2[k]]/sum((calc.mat4[calc.mat.weg.sum > 0.1,calc.mat.zu.sum > 0.1])[,order.y.e2[k]])
        dist.top = cur.line.e2.top - cur.line.e1.top
        dist.bot = cur.line.e2     - cur.line.e1
        grid.y   = c(cur.line.e1.top + dist.top*grid.y.l, (cur.line.e1 + dist.bot*grid.y.l)[length(grid.y.l):1])
        polygon(x.vec, grid.y, col = cur.col, border = cur.col)
        lines(x.vec[1:999], cur.line.e1 + dist.bot*grid.y.l , col = cur.col.l)
        cur.e2[order.y.e2[k]] = cur.line.e2
      }
    }
  }
  cur.y  = 10
  cur.e2 = cur.e2.back
  for (i in 1:length(order.y.e1)){
    j = order.y.e1[i]
    cur.y.t = cur.y
    cur.y   = cur.y - (10-2*box.dist*length(order.y.e1))*grid.y.e1[j] - 2*box.dist
    cur.line     = (calc.mat4[calc.mat.weg.sum > 0.1,][j,])[calc.mat.zu.sum > 0.1]
    cur.line.rel = prop.table(cur.line)
    cur.box.height  = cur.y.t - cur.y - 2*box.dist
    cur.line.e1     = cur.y.t-box.dist
    for (k in 1:length(cur.line)){
      if (cur.line[order.y.e2[k]] > 0.1){
        if (direct == 2 & parties2[calc.mat.zu.sum > 0.1][order.y.e2[k]] %in% party.sel){
          t.cur.col = colv1[calc.mat.weg.sum > 0.1][j]
        } else {
          if (direct == 1 & parties1[calc.mat.weg.sum > 0.1][j] %in% party.sel){
            t.cur.col = colv1[calc.mat.weg.sum > 0.1][j]
          } else {
            t.cur.col = NA
          }
        }
        temp.box.height  = cur.box.height * cur.line.rel[order.y.e2[k]]
        cur.line.e1.top  = cur.line.e1
        cur.line.e1      = cur.line.e1 - temp.box.height
        cur.line.e2.top  = cur.e2[order.y.e2[k]]
        cur.line.e2      = cur.e2[order.y.e2[k]] - cur.e2.height[order.y.e2[k]]*cur.line[order.y.e2[k]]/sum((calc.mat4[calc.mat.weg.sum > 0.1,calc.mat.zu.sum > 0.1])[,order.y.e2[k]])
        dist.top = cur.line.e2.top - cur.line.e1.top
        dist.bot = cur.line.e2     - cur.line.e1
        grid.y   = c(cur.line.e1.top + dist.top*grid.y.l, (cur.line.e1 + dist.bot*grid.y.l)[length(grid.y.l):1])
        polygon(x.vec, grid.y, col = t.cur.col, border = t.cur.col)
        lines(x.vec[1:999], cur.line.e1 + dist.bot*grid.y.l , col =  t.cur.col)
        cur.e2[order.y.e2[k]] = cur.line.e2
      }
    }
  }
}

################################################################################
f1 <- function (mat, mag) {
  matn = mat
  for (i in (1:nrow(mat))) {
    if (sum(mat[i, ] > 0)) {
      matn[i, ] = mat[i, ] * mag[i]/sum(mat[i, ])
    }
  }
  return(matn)
}


################################################################################ 
ipfhk<-function (mat1, mr, mc, max_iter = 2000) 
{
  erg1 = f1(mat1, mr)
  erg2 = t(f1(t(erg1), mc))
  iter = 0
  while (sum(abs(erg1 - erg2)) > 0.001 & iter <= max_iter) {
    erg1 = f1(erg2, mr)
    erg2 = t(f1(t(erg1), mc))
    iter = iter + 1
  }
  if (iter > max_iter) {
    warning("No convergence!")
  }
  return(erg2)
}

################################################################################
ipf = function(mat1, mr, mc, max_iter = 2000, tol = .001){
  erg1 = f1(mat1, mr)
  erg2 = t(f1(t(erg1),mc))
  iter = 0
  while (sum(abs(erg1-erg2)) > tol & iter <= max_iter){
    erg1 = f1(erg2, mr)
    erg2 = t(f1(t(erg1),mc))
    iter = iter + 1
  }
  if(iter > max_iter){
    warning("No convergence!")
  }
  return(erg2)
}

################################################################################
ipf.iter = function(mat1, mr, mc, max_iter = 2000, tol = .001){
  erg1 = f1(mat1, mr)
  erg2 = t(f1(t(erg1),mc))
  iter = 0
  while (sum(abs(erg1-erg2)) > tol & iter <= max_iter){
    erg1 = f1(erg2, mr)
    erg2 = t(f1(t(erg1),mc))
    iter = iter + 1
  }
  if(iter > max_iter){
    warning("No convergence!")
  }
  return(iter)
}

################################################################################
ipf.calc.tab = function(agg.dat, ind.dat, id = "district", elec1 = 10, elec2 = 15, tab = 0.000001, ...){
    if (sum(is.na(match(agg.dat[,id], ind.dat[,id]))) > 0){
       print(paste("Not all ",id,"s could be matched. NA returned.", sep = ""))
       return(NA)
    }
    part1 = colnames(agg.dat)[grep(elec1, colnames(agg.dat))]
    part2 = colnames(agg.dat)[grep(elec2, colnames(agg.dat))]
    part1.part2 = paste(rep(part1, each = length(part2)), rep(part2, length(part1)), sep = ".")
    if (sum(is.na(match(part1.part2, colnames(ind.dat)[colnames(ind.dat) != id]))) > 0){
       print(paste("Not all party combinations could be matched. NA returned.", sep = ""))
       return(NA)
    }
    ind.dat2 = ind.dat[,c(id,part1.part2)]
    ret.dat  = ind.dat2
    for (i in agg.dat[,id]){
       t.agg.dat = agg.dat[agg.dat[,id] == i,]
       t.ind.dat = ind.dat2[ind.dat2[,id] == i,]
       t.ind.tab = matrix(as.numeric(t.ind.dat[,names(t.ind.dat) != id]), nrow = length(part1), ncol = length(part2), byrow = TRUE, dimnames = list(part1,part2))
       t.ind.pro = prop.table(as.table(t.ind.tab))
       ipf.tab   = ipf(mat = t.ind.pro + tab/sum(t.ind.tab),
                                        mr = as.numeric(prop.table(t.agg.dat[part1])), 
                                        mc = as.numeric(prop.table(t.agg.dat[part2])),
                                        tol = 0.01)
       ret.dat[ret.dat[,id] == i, colnames(ret.dat) != id] = as.numeric(t(ipf.tab)*sum(t.agg.dat[part1]))
    }
    return(ret.dat)
}

################################################################################
ipf.calc.iter.tab = function(agg.dat, ind.dat, id = "district", elec1 = 10, elec2 = 15, tab = 0.000001, ...){
  if (sum(is.na(match(agg.dat[,id], ind.dat[,id]))) > 0){
    print(paste("Not all ",id,"s could be matched. NA returned.", sep = ""))
    return(NA)
  }
  part1 = colnames(agg.dat)[grep(elec1, colnames(agg.dat))]
  part2 = colnames(agg.dat)[grep(elec2, colnames(agg.dat))]
  part1.part2 = paste(rep(part1, each = length(part2)), rep(part2, length(part1)), sep = ".")
  if (sum(is.na(match(part1.part2, colnames(ind.dat)[colnames(ind.dat) != id]))) > 0){
    print(paste("Not all party combinations could be matched. NA returned.", sep = ""))
    return(NA)
  }
  ind.dat2 = ind.dat[,c(id,part1.part2)]
  ret.dat  = vector(mode = "numeric", length = nrow(ind.dat2))
  
  for (i in agg.dat[,id]){
    t.agg.dat = agg.dat[agg.dat[,id] == i,]
    t.ind.dat = ind.dat2[ind.dat2[,id] == i,]
    t.ind.tab = matrix(as.numeric(t.ind.dat[,names(t.ind.dat) != id]), nrow = length(part1), ncol = length(part2), byrow = TRUE, dimnames = list(part1,part2))
    t.ind.pro = prop.table(as.table(t.ind.tab))
    ipf.tab   = ipf.iter(mat = t.ind.pro + tab/sum(t.ind.tab), mr = as.numeric(prop.table(t.agg.dat[part1])), 
                         mc = as.numeric(prop.table(t.agg.dat[part2])),
                         tol = 0.01)
    ret.dat[ind.dat2[,id] == i] = ipf.tab
  }
  return(ret.dat)
}

###Constituencies and BESIP respondents in England, Scotland and Wales
besip_indi_stats <- function(data_besip){
  
  # number districts and number of individual data
  n_indi <- nrow(data_besip)
  n_district <- length(unique(data_besip$district))
  
  # min and max voters in one district
  data_besip$numb_voters <- 1
  numb_voter_dist <- tapply(X = data_besip$numb_voters, INDEX = data_besip$district, 
                            sum)
  min_numb_voter_dist <- min(numb_voter_dist)
  max_numb_voter_dist <- max(numb_voter_dist)
  
  return(c(n_district = n_district, n_indi = n_indi, 
           min_numb_voter_dist = min_numb_voter_dist,
           max_numb_voter_dist = max_numb_voter_dist))
  
}
