% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quotation.R
\name{quotation}
\alias{quotation}
\alias{expr}
\alias{enexpr}
\alias{exprs}
\alias{enexprs}
\alias{ensym}
\alias{ensyms}
\alias{quo}
\alias{enquo}
\alias{quos}
\alias{enquos}
\title{Quotation}
\usage{
expr(expr)

enexpr(arg)

exprs(..., .named = FALSE, .ignore_empty = c("trailing", "none",
  "all"), .unquote_names = TRUE)

enexprs(..., .named = FALSE, .ignore_empty = c("trailing", "none",
  "all"), .unquote_names = TRUE, .homonyms = c("keep", "first", "last",
  "error"), .check_assign = FALSE)

ensym(arg)

ensyms(..., .named = FALSE, .ignore_empty = c("trailing", "none",
  "all"), .unquote_names = TRUE, .homonyms = c("keep", "first", "last",
  "error"), .check_assign = FALSE)

quo(expr)

enquo(arg)

quos(..., .named = FALSE, .ignore_empty = c("trailing", "none", "all"),
  .unquote_names = TRUE)

enquos(..., .named = FALSE, .ignore_empty = c("trailing", "none",
  "all"), .unquote_names = TRUE, .homonyms = c("keep", "first", "last",
  "error"), .check_assign = FALSE)
}
\arguments{
\item{expr}{An expression.}

\item{arg}{A symbol representing an argument. The expression
supplied to that argument will be captured instead of being
evaluated.}

\item{...}{For \code{enexprs()}, \code{ensyms()} and \code{enquos()}, names of
arguments to capture without evaluation (including \code{...}). For
\code{exprs()} and \code{quos()}, the expressions to capture unevaluated
(including expressions contained in \code{...}).}

\item{.named}{Whether to ensure all dots are named. Unnamed
elements are processed with \code{\link[=quo_name]{quo_name()}} to build a default
name. See also \code{\link[=quos_auto_name]{quos_auto_name()}}.}

\item{.ignore_empty}{Whether to ignore empty arguments. Can be one
of \code{"trailing"}, \code{"none"}, \code{"all"}. If \code{"trailing"}, only the
last argument is ignored if it is empty. Note that \code{"trailing"}
applies only to arguments passed in \code{...}, not to named
arguments. On the other hand, \code{"all"} also applies to named
arguments.}

\item{.unquote_names}{Whether to treat \code{:=} as \code{=}. Unlike \code{=}, the
\code{:=} syntax supports \code{!!} unquoting on the LHS.}

\item{.homonyms}{How to treat arguments with the same name. The
default, \code{"keep"}, preserves these arguments. Set \code{.homonyms} to
\code{"first"} to only keep the first occurrences, to \code{"last"} to keep
the last occurrences, and to \code{"error"} to raise an informative
error and indicate what arguments have duplicated names.}

\item{.check_assign}{Whether to check for \code{<-} calls passed in
dots. When \code{TRUE} and a \code{<-} call is detected, a warning is
issued to advise users to use \code{=} if they meant to match a
function parameter, or wrap the \code{<-} call in braces otherwise.
This ensures assignments are explicit.}
}
\description{
\Sexpr[results=rd, stage=render]{rlang:::lifecycle("stable")}

Quotation is a mechanism by which an expression supplied as
argument is captured by a function. Instead of seeing the value of
the argument, the function sees the recipe (the R code) to make
that value. This is possible because R \link[=is_expr]{expressions} are
representable as regular objects in R:
\itemize{
\item Calls represent the action of calling a function to
compute a new value. Evaluating a call causes that value to be
computed. Calls typically involve symbols to reference R objects.
\item Symbols represent the name that is given to an object in a
particular context (an \link[=env]{environment}).
}

We call objects containing calls and symbols \link[=is_expr]{expressions}.
There are two ways to create R expressions. First you can \strong{build}
calls and symbols from parts and pieces (see \code{\link[=sym]{sym()}}, \code{\link[=syms]{syms()}} and
\code{\link[=call2]{call2()}}). The other way is by \emph{quotation} or \emph{quasiquotation},
i.e. by intercepting an expression instead of evaluating it.
}
\section{User expressions versus your expressions}{


There are two points of view when it comes to capturing an
expression:
\itemize{
\item You can capture the expressions supplied by \emph{the user} of your
function. This is the purpose of \code{ensym()}, \code{enexpr()} and
\code{enquo()} and their plural variants. These functions take an
argument name and capture the expression that was supplied to
that argument.
\item You can capture the expressions that \emph{you} supply. To this end
use \code{expr()} and \code{quo()} and their plural variants \code{exprs()} and
\code{quos()}.
}
}

\section{Capture raw expressions}{

\itemize{
\item \code{enexpr()} and \code{expr()} capture a single raw expression.
\item \code{enexprs()} and \code{exprs()} capture a list of raw expressions
including expressions contained in \code{...}.
\item \code{ensym()} and \code{ensyms()} are variants of \code{enexpr()} and
\code{enexprs()} that check the captured expression is either a string
(which they convert to symbol) or a symbol. If anything else
is supplied they throw an error.
}

In terms of base functions, \code{enexpr(arg)} corresponds to
\code{base::substitute(arg)} (though that function also features complex
substitution semantics) and \code{expr()} is like \code{\link[=quote]{quote()}} (and
\code{\link[=bquote]{bquote()}} if we consider unquotation syntax). The plural variant
\code{exprs()} is equivalent to \code{\link[base:alist]{base::alist()}}. Finally there is no
function in base R that is equivalent to \code{enexprs()} but you can
reproduce its behaviour with \code{eval(substitute(alist(...)))}.
}

\section{Capture expressions in quosures}{


\code{quo()} and \code{enquo()} are similar to their \code{expr} counterparts but
capture both the expression and its environment in an object called
a quosure. This wrapper contains a reference to the original
environment in which that expression was captured. Keeping track of
the environments of expressions is important because this is where
functions and objects mentioned in the expression are defined.

Quosures are objects that can be evaluated with \code{\link[=eval_tidy]{eval_tidy()}} just
like symbols or function calls. Since they always evaluate in their
original environment, quosures can be seen as vehicles that allow
expressions to travel from function to function but that beam back
instantly to their original environment upon evaluation.

See the \link{quosure} help topic about tools to work with quosures.
}

\section{Quasiquotation}{


All quotation functions in rlang have support for \link[=quasiquotation]{unquoting operators}. The combination of quotation and
unquotation is called \emph{quasiquotation}.

Unquotation provides a way to refer to variables during quotation.
Variables are problematic when quoting because a captured
expression is essentially a constant, just like a string is a
constant. For instance in all the following cases \code{apple} is a
constant: \code{~apple}, \code{"apple"} and \code{expr(apple)}. Unquoting allows
you to introduce a part of variability within a captured
expression.
\itemize{
\item In the case of \code{enexpr()} and \code{enquo()}, unquoting provides an
escape hatch to the users of your function that allows them to
manipulate the expression that you capture.
\item In the case of \code{expr()} and \code{quo()}, quasiquotation lets you
build a complex expressions where some parts are constant (the
parts that are captured) and some parts are variable (the parts
that are unquoted).
}

See the \link{quasiquotation} help topic for more about this as well as
\href{https://adv-r.hadley.nz/quasiquotation.html}{the chapter in Advanced R}.
}

\examples{
# expr() and exprs() capture expressions that you supply:
expr(symbol)
exprs(several, such, symbols)

# enexpr() and enexprs() capture expressions that your user supplied:
expr_inputs <- function(arg, ...) {
  user_exprs <- enexprs(arg, ...)
  user_exprs
}
expr_inputs(hello)
expr_inputs(hello, bonjour, ciao)

# ensym() and ensyms() provide additional type checking to ensure
# the user calling your function has supplied bare object names:
sym_inputs <- function(...) {
  user_symbols <- ensyms(...)
  user_symbols
}
sym_inputs(hello, "bonjour")
## sym_inputs(say(hello))  # Error: Must supply symbols or strings
expr_inputs(say(hello))


# All these quoting functions have quasiquotation support. This
# means that you can unquote (evaluate and inline) part of the
# captured expression:
what <- sym("bonjour")
expr(say(what))
expr(say(!!what))

# This also applies to expressions supplied by the user. This is
# like an escape hatch that allows control over the captured
# expression:
expr_inputs(say(!!what), !!what)


# Finally, you can capture expressions as quosures. A quosure is an
# object that contains both the expression and its environment:
quo <- quo(letters)
quo

get_expr(quo)
get_env(quo)

# Quosures can be evaluated with eval_tidy():
eval_tidy(quo)

# They have the nice property that you can pass them around from
# context to context (that is, from function to function) and they
# still evaluate in their original environment:
multiply_expr_by_10 <- function(expr) {
  # We capture the user expression and its environment:
  expr <- enquo(expr)

  # Then create an object that only exists in this function:
  local_ten <- 10

  # Now let's create a multiplication expression that (a) inlines
  # the user expression as LHS (still wrapped in its quosure) and
  # (b) refers to the local object in the RHS:
  quo(!!expr * local_ten)
}
quo <- multiply_expr_by_10(2 + 3)

# The local parts of the quosure are printed in colour if your
# terminal is capable of displaying colours:
quo

# All the quosures in the expression evaluate in their original
# context. The local objects are looked up properly and we get the
# expected result:
eval_tidy(quo)
}
