/*====================================================================
an_resp_burden : PERFORMS RESPONDENT BURDEN PAPER ANALYSIS 

Project:       Respondent Burden Paper
Author:        Brendan Read
Organisation:  ISER, University of Essex
----------------------------------------------------------------------
Creation Date:    2016/10/19
Modification Date: 2018/12/12
Do-file version:    07

Input:

 RQ1.dta - Analytical data set for RQ1
  RQ2_sub.dta - Analytical data set for RQ2 subjective analyses
  RQ2_obj.dta - Analytical data set for RQ2 objective analyses
  RQ3.dta - Analytical data set for RQ3
  RQ4_sub.dta - Analytical data set for RQ4 subjective analyses
  RQ4_obj.dta - Analytical data set for RQ4 objective analyses

====================================================================*/

*Set-up

local data "[data directory]"  //Local macro for the directory containing the data
cd `data'

/* RQ1 */

*Install user written commands
  
  /*Commands for if packages are missing
  capture net from http://www.komkon.org/~tacik/stata/
  capture net install polychoric.pkg*/
  
*Load the data file

  use "`data'\RQ1.dta", clear

 *Establish macros
  
  local objective "avg_duration tot_duration appuses"
  local subjective "two_likely two_effort two_interest two_difficulty"
 
*Save means and standard deviations for factor analysis
 
  tabstat `subjective' `objective', stat(mean) save
  matrix mean = r(StatTotal)
  mat list mean

  tabstat `subjective' `objective', stat(sd) save
  matrix sd = r(StatTotal)
  mat list sd

/*Polychoric correlations - All*/

  polychoric `subjective' `objective'

  matrix poly = r(R)
  
  local p=7
  local n=223
  scalar df=(1/2)*`p'*(`p'-1)
  scalar chi2 = -[`n'-1-(1/6)*(2*`p'+5)]*ln(det(poly))
  scalar pval=chi2tail(df, chi2)
  
  di chi2
  di df
  di pval
  
  factormat poly, n(223) sds(sd) means(mean)
  rotate
  
  estat kmo

/* RQ2 */

/*Subjective burden*/

*Install user written package (sq.ado by Kohler et al.)

 *Commands for if package is missing 
  capture net from http://www.stata-journal.com/software/sj6-4
  capture net install st0111.pkg
*/
*Load the RQ2 subjective dataset

  use "`data'\RQ2_sub.dta", clear

*Sequence analysis

*New ordering

gen ord_1 = revdiff * 1000 if week == 1
gen ord_2 = revdiff * 100 if week == 2
gen ord_3 = revdiff * 10 if week == 3
gen ord_4 = revdiff * 1 if week == 4

replace ord_1 = 0 if ord_1 == .
replace ord_2 = 0 if ord_2 == .
replace ord_3 = 0 if ord_3 == .
replace ord_4 = 0 if ord_4 == .

gen order_sum = ord_1 + ord_2 + ord_3 + ord_4
bysort pidp: egen new_order = total(order_sum) 

*Sequence plot
  sqset revdiff pidp week
  sqindexplot, order(new_order)

*Fixed effects model

  xtset pidp
  xtreg revdiff week, fe vce(cluster i_psu)
  xtreg revdiff week if revdiff!=5, fe vce(cluster i_psu)
  
/*Objective burden*/

*Load the dataset

  use "`data'\RQ2_obj.dta", clear

*App uses

  xtset pidp
 
  xtreg duration appuse_no, fe vce(cluster i_psu)
  local aua = _b[_cons]
  local aub = _b[appuse_no]

*Receipts scanned  
  
  xtreg duration activity_no if activitytype==1, fe vce(cluster i_psu)
  local rsa = _b[_cons]
  local rsb = _b[activity_no]
  summ activity_no if activitytype == 1 //Calculate range for graphing
  local rsm = r(max)

*Purchases without receipt
  
  xtreg duration activity_no if activitytype==2, fe vce(cluster i_psu)
  local pwa = _b[_cons]
  local pwb = _b[activity_no]
  summ activity_no if activitytype == 2 //Calculate range for graphing
  local pwm = r(max)

*Nothing bought

  xtreg duration activity_no if activitytype==3, fe vce(cluster i_psu)
  local nba = _b[_cons]
  local nbb = _b[activity_no]
  summ activity_no if activitytype == 3 //Calculate range for graphing
  local nbm = r(max)
  
*Graph  
  
  twoway function y = `aua' + (`aub'*x), range(appuse_no) || /// 
  function y = `rsa' + (`rsb'*x), range(1 `rsm') || ///
  function y = `pwa' + (`pwb'*x), range(1 `pwm') || ///
  function y = `nba' + (`nbb'*x), range(1 `nbm')

* Use RQ3 Data for additional RQ2 analysis

  use "`data'\RQ3.dta", clear

* Analysis of changes in proportion of types of app use  
  
  xtset pidp
  xtreg rsprop day, fe
  xtreg pwprop day, fe
  xtreg nbprop day, fe

  /* RQ3 */

*Load data

  use "`data'\RQ3.dta", clear

*Dropout
stset day, id(pidp) failure(dropout=1)
stdescribe
stcox avgdur pwprop nbprop, vce(cluster i_psu)

  
*Day missed
stset day, id(pidp) failure(daywithout=1)
stdescribe
stcox avgdur pwprop nbprop, vce(cluster i_psu)

*Day missed, repeated spells
stset day, id(pidp) failure(daywithout=1) exit(dropout=1)
stdescribe
stcox avgdur pwprop nbprop, vce(cluster i_psu)

/* RQ4 */

 **SUBJECTIVE BURDEN
  
  use "`data'\RQ4_sub.dta", clear

*Bivariate - Chi2  
  
 svyset i_psu

  foreach sub of varlist eop_likely eop_effort eop_interest mdifficulty {
    foreach var of varlist disill bankbal i_sex i_hiqual_dv device i_scphoto ///
    i_scbank i_scinstal i_scwill2 i_scwill5 finbud timeconstraint incent ///
	unconditional poverty{
     svy: tabulate `var' `sub', pear
    }
  }


*Likelihood
 	
svy: logit two_likely incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage, or 

svy: logit two_likely incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage 

*Effort  
svy: logit two_effort incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage, or

svy: logit two_effort incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage
  
*Interest  
svy: logit two_interest incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage, or 

svy: logit two_interest incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage

svy: logit two_interest incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage i_pdvage#i_pdvage 

*Difficulty  
svy: logit two_difficulty incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage, or 

svy: logit two_difficulty incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage 


**OBJECTIVE BURDEN

  use "`data'\RQ4_obj.dta", clear
  
  svyset i_psu

*Incentive treatment

  svy: mean duration, over(incent)
  lincom [duration]0 - [duration]1
  
*Additional incentive  

  svy: mean duration, over(unconditional)
  lincom [duration]_subpop_1 - [duration]_subpop_2
   
*Uses device for taking photos
  svy: mean duration, over(i_scphoto)
  lincom [duration]No - [duration]Yes

*Uses device for online banking
  svy: mean duration, over(i_scbank)
  lincom [duration]No - [duration]Yes

*Uses device to install apps
  svy: mean duration, over(i_scinstal)
  lincom [duration]No - [duration]Yes
 
*Willingness to download app
  svy: mean duration, over(two_app)
  lincom [duration]0 - [duration]1

*Willingness to use camera
  svy: mean duration, over(two_cam)
  lincom [duration]0 - [duration]1
  
*Frequency of checking bank balance
  svy: mean duration, over(bankbal)
  lincom [duration]_subpop_1 - [duration]_subpop_2

*Keeps a budget
  svy: mean duration, over(finbud)
  lincom [duration]_subpop_1 - [duration]_subpop_2
  
*Poverty threhold
  svy: mean duration, over(poverty)
  lincom [duration]0 - [duration]1

*Time constrained
  svy: mean duration, over(timeconstraint)
  lincom [duration]0 - [duration]1

*Level of education

  svy: mean duration, over(i_hiqual_dv)
  lincom [duration]_subpop_1 - [duration]_subpop_2

*Disability/illness
    
  svy: mean duration, over(disill)
  lincom [duration]_subpop_1 - [duration]_subpop_2
  
*Gender
    
  svy: mean duration, over(i_sex)
  lincom [duration]male - [duration]female 
  
*Mixed effects regression model
mixed duration incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage ///
i.activitytype|| pidp:, vce(cluster i_psu)

mixed duration incent unconditional i_scphoto i_scbank i_scinstal two_app two_cam ///
bankbal finbud poverty timeconstraint i_hiqual_dv disill i_sex i_pdvage ///
i.activitytype|| pidp:, vce(cluster pidp)
