/*
Paper: 		Memory Gabs in the American Time Use Survey. Are Respondents Forgetful or is There More to it?
Authors: 	Antje Kirchner ab, Robert F Belli b, Ana Lucía Córdova-Cazar c, Caitlin E Deal d
			a RTI International, b University of Nebraska-Lincoln, c Universidad San Francisco de Quito, d AIR
Date: 		26 August 2018
File: 		Data Cleaning
*/



capture log close
version 15.1
clear all
set more off

//Set Paths
	global pathdta "SET YOUR DIRECTORY\dta"
	global pathlog "SET YOUR DIRECTORY\log"
	global pathplot "SET YOUR DIRECTORY\plot"

	local today=string(date("`c(current_date)'","DMY"),"%tdCCYYNNDD")
	
log using "$pathlog/1_atus_`today'.smcl", replace 

***********************************************
*Read in public use data (Retrieved from https://www.bls.gov/tus/datafiles_2010.htm.)
*1) ATUS 2010 ATUS-CPS File
	use "$pathdta/orig/Atuscps_2010.dta", clear 
	
	sum prtage, d
	gen age= prtage - r(mean)
	sum prtage if tratusr==1
	
		lab var age "R. Age in years (centered)"
		sum age
		
		gen age_sq=age^2
		label var age_sq "R. Age in years squared"
	
	rename pesex female
		lab var female "R. Female (ref. Male)"
		tab female, m
 	
	gen edu = .
		replace edu = 0 if inlist(peeduca, 31,32,33,34,35,36,37,38,39)
		replace edu = 1 if inlist(peeduca, 40,41,42)
		replace edu = 2 if inlist(peeduca, 43)
		replace edu = 3 if inlist(peeduca, 44,45,46)
		replace edu = 4 if inlist(peeduca, -1)
		lab def edu 0 "High school or less" 1 "Some college; Associate degree" 2 "Bachelor" 3 "Graduate" 4 "No information"
		lab val edu edu
		lab var edu "R: Education (ref. High school or less)"
		tab peeduca edu, m
	
	tab ptdtrace
	gen race = .
		replace race = 0 if ptdtrace == 1
		replace race = 1 if ptdtrace != 1
		lab def race 0 "White" 1 "Non-White or multiple"
		lab val race race
		lab var race "R: Race (ref. White)"
		tab ptdtrace race, m
		
	gen metro = .
		replace metro = 0 if gtmetsta == 1
		replace metro = 1 if gtmetsta == 2
		replace metro = 2 if gtmetsta == 3
		lab def metro 0 "Metropolitan" 1 "Non-metropolitan" 2 "Not identified"
		lab val metro metro
		lab var metro "R: Metropolitan status (2000 definitions)"
		tab gtmetsta metro , m
	
	gen empl = .
		replace empl = 0 if inlist(pemlr, 5,6,7)
		replace empl = 1 if inlist(pemlr, 3,4)
		replace empl = 2 if inlist(pemlr, 1,2)
		lab def empl 0 "Not in labor force" 1 "Unemployed" 2 "Employed"
		lab val empl empl	
		lab var empl "R: Labor force status"
		tab pemlr empl, m
		tab empl, m
	
	gen marstat = .
		replace marstat = 0 if pemaritl == 1
		replace marstat = 1 if pemaritl != 1 & pemaritl!=-1
		lab def marstat 0 "Married" 1 "Other"
		lab val marstat marstat
		lab var marstat "R. Marital status (ref. Married)"
		tab pemaritl marstat, m
		tab marstat, m
	
	rename prnmchld childnum  
		lab var childnum "R. Number of own children < 18 years of age"  
		replace childnum=. if childnum==-1
		sum childnum
		tab childnum, m
	
	tab hefaminc hryear4
	tab hefaminc hxfaminc, m
	
		gen famincdk=1 if inlist(hxfaminc, 22, 42)
		replace famincdk=0 if famincdk==. & hefaminc!=-1
		replace famincdk=2 if hefaminc==-1
		gen famincref=1 if inlist(hxfaminc, 23, 43)	
		replace famincref=0 if famincref==. & hefaminc!=-1
		replace famincref=2 if hefaminc==-1
		gen famincdkref=1 if inlist(hxfaminc, 22, 42)	
		replace famincdkref=2 if inlist(hxfaminc, 23, 43)	
		replace famincdkref=0 if hxfaminc==0
		replace famincdkref=3 if hefaminc==-1 | inlist(hxfaminc, 21, 41)
		
		lab def famincdk 0 "No dk" 1 "DK" 2 "No information"
		lab def famincref 0 "No ref" 1 "REF" 2 "No information"
		lab def famincdkref 0 "Valid entry" 1 "DK" 2 "REF" 3 "No information"
		
		lab val famincdk famincdk
		lab val famincref famincref
		lab val famincdkref famincdkref
		
		tab1 faminc* 
	
		sum hefaminc if hefaminc>=1, d
		tab hefaminc if hefaminc>=1
		
		recode hefaminc (1/8 = 0 "Less than $30,000") (9/13 = 1 "$30,000 to $74,999") (14/max = 2 "$75,000 and up") (-1=3 "No information"), gen(faminc)
		clonevar faminc2 = faminc
		replace faminc2 = 0 if hufaminc>=1 & hufaminc<=8 & faminc==3
		replace faminc2 = 1 if hufaminc>=9 & hufaminc<=13 & faminc==3
		replace faminc2 = 2 if hufaminc>=14 & hufaminc<=16 & faminc==3
		gen famincblank=0
		replace famincblank=1 if hefaminc==3
		
		clonevar famincref2=famincref
		replace famincref2=0 if inlist(faminc2, 0,1,2) & famincref!=1
		tab famincref2
		
		label var faminc "R. Family income"
		label var faminc2 "R. Family income (he and hu combined)"
		label var famincblank "R. Family income blank in hefaminc"
		label var famincdkref "R. Family income dk, ref, or no info"
		label var famincdk "R. Family income dk"
		label var famincref "R. Family income ref (Income 2010)"
		label var famincref2 "R. Family income ref (Income 2009 & 2010 combined)"
		
	recode hetenure (1=1 "Home owner") (2/3=0 "Rent / Other"), gen(own)
		label var own "R. Home owner"
	
	recode pehspnon (1=1 "Hispanic") (2=0 "Non-Hispanic"), gen(ethn)
		label var ethn "R. Hispanic"
		
	keep if tratusr==1
		
	save "$pathdta/use_atus_demographics_`today'.dta", replace
	
*2) ATUS 2010 Respondent File
	use "$pathdta/orig/Atusresp_2010.dta", clear		
		
	rename tudiaryday diaryday
		lab var diaryday "R: Diary day"		
		tab diaryday, m
	
	rename trholiday holiday
		lab var holiday "R: Diary day was a holiday"
		tab  holiday, m
		
	rename trhhchild child_pres
	tab child_pres, nol
		recode child_pres (1=1 "yes") (2=0 "no"), gen(child_presrec)
		tab child_pres child_presrec
		lab var child_presrec "R. Presence of child in hh <18"
	
	merge 1:1 tucaseid using "$pathdta/use_atus_demographics_`today'.dta"
		tab _merge
		keep if _merge==3
		drop _merge
	
	keep tucaseid tufinlwgt age age_sq prtage female edu race ethn metro empl marstat faminc* own child_pres* childnum diaryday holiday
	sort tucaseid
	
	destring tucaseid, gen(ntucaseid)
	drop tucaseid
	rename ntucaseid tucaseid
	
	save "$pathdta/use_atus_demographics_`today'.dta", replace
	
*3)	ATUS 2010 Call History File
	use "$pathdta/orig/atuscall_2010.dta", clear
	/* Stata will not process these values.
		label define labeltrfnlcll
		-1 "Blank"
		-2 "Don't Know"
		-3 "Refused"
		1.001 "Complete interview"
		2.001 "Sufficient partial"
		20.002 "Not eligible: other type of living quarters"
		20.011 "Not eligible: designated person underage"
		20.022 "Not eligible: designated person in Armed Forces"
		20.015 "Not eligible: designated person not a household member"
		20.007 "Not eligible: vacant"
		20.005 "Not eligible: not used as a regular residence"
		21.003 "Not eligible: designated person moved out"
		21.002 "Other: designated person absent, ill, or hospitalized - unavailable through closeout"
		21.001 "Other: designated person institutionalized, unavailable through closeout"
		22.002 "Unknown eligibility: sample unit not found/unreached"
		24.001 "Other: language barrier"
		25.001 "Other: unconverted hearing barrier"
		172.001 "Not eligible: removed from sample"
		172.002 "Other: invalid input"
		176.001 "Refusal: Congressional case"
		179.001 "Refusal: hostile breakoff"
		181.001 "Refusal: refusal by designated person or gatekeeper"
		182.001 "Refusal: manual override by supervisor"
		183.001 "Unknown eligibility: exceeded unproductive call maximum"
		185.001 "Sufficient partial with planned callback"
		186.001 "Refusal: pre-refusal based on explicit refusal or hostile breakoff"
		188.001 "Noncontact: incomplete callbacks"
		188.003 "Not eligible: temporarily unavailable (institutionalized)"
		188.002 "Noncontact: temporarily unavailable (absent, ill, or hospitalized)"
		191.001 "Other: unresolved language barrier"
		192.001 "Other: hearing barrier"
		193.001 "Unknown eligibility: privacy detectors"
		194.001 "Noncontact: never contacted, confirmed number"
		195.001 "Unknown eligibility: never contacted, unconfirmed number"
		199.001 "Unknown eligibility: never tried, no telephone number household" */
	
	//derive interviewer cooperation rate, ever refused, etc.
		format %16.0g tucaseid
		sort tucaseid tuattmdate
		gen long dispo=trfnlcll
		tab dispo
		tab trfnlcll
		
		by tucaseid: gen attempts = _N
		label var attempts "R. Number of contact attempts"
		tab attempts
		
		gen everr = 1 if dispo==179 | dispo==181 | dispo==186
		replace everr = 0 if everr==.
		tab everr
		by tucaseid: egen everref=max(everr)
		label var everref "R. Initial refusal"
		tab everref
		drop everr
		
		sort tuhintid tucaseid tuattmdate
		gen cont = 1 if inlist(dispo, 1,2,24,25,179,181,185,186,191,192)
		by tuhintid: gen conta=sum(cont)
		by tuhintid: egen contact=max(conta)
		drop cont conta
		
		gen comp = 1 if inlist(dispo, 1,2)
		by tuhintid: gen compl=sum(comp)
		by tuhintid: egen completes=max(compl)
		drop comp compl
		
		label var completes "I. Experience"
	
		gen iverCR=completes/contact
		label var iverCR "I. Cooperation rate"
		sum iverCR
		gen centiverCR=iverCR-r(mean)
		label var centiverCR "I. Cooperation rate (centered)"
		
		egen intid=group(tuhintid)
		label var intid "I. Interviewer ID"
		tab intid
		
	//keep only completes & merge
		keep if dispo==1 | dispo==2						
		
		merge 1:1 tucaseid using "$pathdta/use_atus_demographics_`today'.dta"
			tab _merge
			keep if _merge==3
			drop _merge
			
		merge 1:1 tucaseid using "$pathdta/orig/atuscase.dta"		// ATUS 2010 Case History File
			tab _merge
			keep if _merge==3
			drop _merge
			
		tab tucpsdp, nol
		recode tucpsdp (1=0 "same") (2=1 "different"), gen(ATUSrespondentNOTCPS)
	
	keep tucaseid tufinlwgt tucpsdp age age_sq prtage female edu race ethn metro empl marstat faminc* own child_pres* childnum diaryday holiday intid attempts everref iverCR centiverCR completes tr2intst tuavgdur tucpsdp tudqual2 tuincent tuintdqual tulngskl tutotactno ATUSrespondentNOTCPS
	sort tucaseid
	save "$pathdta/use_atus_demographics_`today'.dta", replace
	

//	ATUS 2010 Activity File
	use "$pathdta/orig/atusact_2010.dta", clear			
	format %16.0g tucaseid
		
		tab trcode
		gen forget=(trcode==500106)
		gen insufficient=inlist(trcode,500106,500103,500107)
		gen refusal=(trcode==500105)
		tab1 forget insufficient refusal, m
		
		gen durforget=tuactdur if forget==1
		gen durinsufficient=tuactdur if insufficient==1
		gen durrefusal=tuactdur if refusal==1
		
		collapse (sum) forget insufficient refusal (mean) dur*, by(tucaseid)
		tab1 forget insufficient refusal, m
		sum dur*
		replace durforget=0 if durforget==.
		sum durforget
		
		label var forget "R. ATUS enduring memory gap"
		label var insufficient "R. ATUS insufficient detail"
		label var refusal "R. ATUS refusal"
	
	merge 1:1 tucaseid using "$pathdta/use_atus_demographics_`today'.dta"
		tab _merge
		keep if _merge==3
		drop _merge
		sort tucaseid
		
	save "$pathdta/use_atus_demographics_`today'.dta", replace
	
**************************************
*Audittrails (ICPSR deposit number 37282) 
*Generate number of memory gaps per respondent, number of who and where changes per respondent etc.

*Step 1, read in demographics, recode & merge in audit trail info (only to do once)
	/*
	import delimited "$pathdta/orig/ATUS_ActionLevelData.csv", asdouble clear 
	
	//labeling of some variables
		#delimit ;
		lab def action 
			-8	"Not Applicable"
			0 	"Activity Type"
			1	"Time Type"
			2	"Minute Duration"
			3	"Hour Duration"
			4	"Stop Time"
			5	"Who"
			6	"Where"
			7	"Where Specified (Verbatim Where)"
			8	"Insert Activity"
			9	"Delete Activity"
			10	"Undo Insert Activity"
			11	"Eat Check Screen"
			12	"Delete Prompt"
			13	"Duration Too Long Prompt"
			14	"Stop Time Too Long Prompt"
			15	"End of Day Prompt"
			16	"Subsequent Time Affected Prompt"
			17	"Missing Activity Prompt"
			18	"Missing Who Prompt"
			19	"Missing Where Prompt"
			20	"Missing Where Specified Prompt"
			21	"Missing Time Prompt"
			22	"Missing Eating Prompt"
			23	"Work Prompt"
			24	"Illegal Who Value Prompt"
			25	"Undo Insert Prompt"
			26	"Child Care Prompt"
			27	"Unknown Prompt Type"
			1095 "Continue to Household Roster"
			.a  "not applicable";
		
		lab def codedcontent 
			-8 "Not Applicable"
			0 "Missing Activity"
			1 "Sleeping"
			2 "Grooming Self"
			3 "Watching TV"
			4 "Working at Main Job"
			5 "Working at Other Job"
			6 "Preparing Meal or Snacks"
			7 "Eating and Drinking"
			8 "Cleaning Kitchen"
			9 "Doing Laundry"
			10 "Grocery Shopping"
			11 "Attending Religious Service"
			12 "Paying Household Bills"
			13 "Verbatim Activity"
			30 "Don’t Know or Can’t Remember Activity"
			31 "Refusal or None of Your Business Activity"
			100 "Missing Where"
			101 "Respondent’s Home or Yard"
			102 "Respondent’s Workplace"
			103 "Someone Else’s Home"
			104 "Restaurant or Bar"
			105 "Place of Worship"
			106 "Grocery Store"
			107 "Other Store or Mall"
			108 "School"
			109 "Outdoors Away from home"
			110 "Library"
			111 "Other Place (Specify)"
			112 "Driver of Car, Truck, or Motorcycle"
			113 "Passenger of Car, Truck, or Motorcycle"
			114 "Walking"
			115 "Bus"
			116 "Subway or Train"
			117 "Bicycle"
			118 "Boat or Ferry"
			119 "Taxi or Limousine Service"
			120 "Airplane"
			121 "Other Mode of Transportation (Specify)"
			130 "Bank"
			131 "Gym or Health Club"
			132 "Post Office"
			10001 "Stop Time"
			10002 "Duration"
			1095 "Continue to Household Roster"
			.a "not applicable";

		lab def whocontent
			-8 "Not Applicable"
			1000 "Missing Who"
			1001 "Alone"
			1002 "Family Member 2"
			1003 "Family Member 3"
			1004 "Family Member 4"
			1005 "Family Member 5"
			1006 "Family Member 6"
			1007 "Family Member 7"
			1008 "Family Member 8"
			1009 "Family Member 9"
			1010 "Family Member 10"
			1050 "All Household Members"
			1051 "Parents"
			1052 "Other Non-household Family Members under 18"
			1053 "Other Non-household Family Members 18 or Older"
			1054 "Friends"
			1056 "Neighbors or Acquaintances"
			1057 "Other Non-household Children under 18"
			1058 "Other Non-household Adults 18 or Older"
			1059 "Boss or Manager"
			1060 "People Whom I Supervise"
			1061 "Coworkers"
			1062 "Customers"
			1095 "Continue to Household Roster"
			.a "not applicable"
			.b "missing";
	 #delimit cr
	
	lab val action action
	lab val codedcontent codedcontent
	
	replace whocontent=".a" if whocontent=="-8"
	split whocontent, p("-")
	replace whocontent="-8" if whocontent==".a"
	destring whocontent1-whocontent10, replace
	lab val whocontent1-whocontent10 whocontent
	
	mvdecode _all, mv(-8=.a \ -9=.b)
	format %16.0g tucaseid
	
	gen double timebegan2=clock(timebegan, "MDYhms")
	order timebegan2, after(timebegan)
	format %tc timebegan2
	
	gen double timeended2=clock(timeended, "MDYhms")
	order timeended2, after(timeended)
	format %tc timeended2
	
	drop timebegan timeended
	rename timebegan2 timebegan
	rename timeended2 timeended
	
	save "$pathdta/ATUS_ActionLevelData.dta", replace		
	*/
	
	use "$pathdta/ATUS_ActionLevelData.dta", clear
	
		gen id=_n
				
	//generate indicator for time elapsed overall before anything is dropped
		sort tucaseid tuactivity_n timebegan actionnum
		sum timeelapsed
		by tucaseid tuactivity_n: gen total=sum(timeelapsed)
		by tucaseid tuactivity_n: egen activitytime=max(total)
		
		sum total*
		drop total
		
		sum activitytime, d										
		replace activitytime=r(p99) if activitytime>=r(p99)		
		sum activitytime	

	//total interview time ATUS
		by tucaseid: egen double interviewstart=min(timebegan)
		format %tc interviewstart
		by tucaseid: egen double interviewend=max(timeended)
		format %tc interviewend
		gen double intdur = minutes(interviewend-interviewstart)
		
	//drop unnecessary entries	
		drop if tucaseid==.b 				//drop cases who are not in the 2010 public release file (e.g., cases with five activities or less etc.)
		drop if tuactivity_n<0				//drop deleted entries (0.34%)
		drop if dataentered==0				//drop all purely navigational activities
		
		*check for duplicates
		duplicates tag audittrail tucaseid activityseqnum rownum tuactivity_n action timebegan timeended timeelapsed withres codedcontent numericcontent whocontent dataentered imputedtimebegan imputedtimeended matchedinserts whocontent1 whocontent2 whocontent3 whocontent4 whocontent5 whocontent6 whocontent7 whocontent8 whocontent9 whocontent10, gen(dup)
		tab dup		//none
		drop dup
		sort id
			
	//drop a few spells that are not needed  ask AE == interviewers just passing through them without editing == safe to drop
		tab action
		keep if inlist(action, 0,5,6)		//keep activity type, who, where
		tab action
		
		drop if regexm(whocontent, "1000")  // drop missing who information (n=119) --> sensitivity analyses shows no difference when we retain those cases
		
		sort tucaseid tuactivity_n timebegan actionnum	
		order timebegan-withres, after(whocontent10)
		drop numericcontent
		order whocontent1-whocontent10, after(whocontent)
		
	//generate final activity type...
		sort tucaseid tuactivity_n action timebegan actionnum
		by tucaseid tuactivity_n action: gen action_n=_n
		by tucaseid tuactivity_n action: gen action_fin=action_n if _n==_N
		by tucaseid tuactivity_n action: egen action_final=max(action_fin)
		tab1 action_fin action_final
		order action_n action_fi*
		drop action_fin
	
	//generate final number of memory gaps 
		gen memgap1=1 if (codedcontent==30) & action_final==action_n & action==0 
		tab memgap1 if action==0
		tab memgap dataentered if action==0, m
		by tucaseid: egen memg=sum(memgap1)
		by tucaseid: egen memgap=max(memg)
		tab memgap1
		order memgap*
		
		drop memgap1 memg
		tab memgap
		
		gen d_memgap=(memgap>=1)
		tab d_memgap memgap, m
		
		
	//generate number of temporary memory gaps 
		gen memgap1_any=1 if (codedcontent==30) & action_final!=action_n  & action==0 	
		tab memgap1_any if action==0
		by tucaseid: egen memg_any=sum(memgap1_any)
		by tucaseid: egen memgap_any=max(memg_any)
		tab memgap1_any
		order memgap1_any memg_any memgap_any
		
		drop memgap1_any memg_any 
		tab memgap_any
		
		gen d_memgap_any=(memgap_any>=1)
		tab d_memgap_any memgap_any, m
		
		corr memgap memgap_any
		
	//generate number of who changes ("ever" in mention of one activity and then sum up)
		duplicates tag tucaseid tuactivity_n action whocontent1-whocontent10, gen(dup)
		replace dup=dup+1
		
		gen whoch=1 if (action_n==action_fin) & dup!=action_fin & action==5 
		tab whoch
		by tucaseid: egen whochg=sum(whoch)
		order dup whoch*
		
		drop whoch dup
		tab whochg				
				
	//generate number of where changes ("ever" in mention of one activity and then sum up)
		duplicates tag tucaseid tuactivity_n action codedcontent, gen(dup)		
		replace dup=dup+1
		
		gen wherech=1 if (action_n==action_fin) & dup!=action_fin  & action==6 
		tab wherech
		by tucaseid: egen wherechg=sum(wherech)
		order dup wherech*
		
		drop wherech dup
		tab wherechg
		
	//generate number of who changes (sum)
		sort tucaseid tuactivity_n action timebegan actionnum
		
		by tucaseid tuactivity_n action: gen whoch= 1 if whocontent[_n]!=whocontent[_n-1] & action==5 & action_final>=2 & action[_n]==action[_n-1]
		tab whoch
		by tucaseid: egen whochg_any=sum(whoch)
		order  whoch whocontent action
		
		drop whoch
		tab whochg_any whochg, m
		
	//generate number of where changes (sum) 
		by tucaseid tuactivity_n action: gen wherech= 1 if codedcontent[_n]!=codedcontent[_n-1] & action==6 & action_final>=2 & action[_n]==action[_n-1]
		tab wherech
		by tucaseid: egen wherechg_any=sum(wherech)
		order wherech codedcontent action
		
		drop wherech
		tab wherechg_any wherechg, m
		
		order memgap whochg whochg_any wherechg wherechg_any, after(tuactivity_n)
		
	//aggregate
		collapse 	(mean) audittrail memgap_any d_memgap_any whochg_any wherechg_any activitytime intdur  ///
					(max) tuactivity_n matchedinserts, by(tucaseid)		
					
		sum intd, d
		gen intdur_cens=intdur
		replace intdur_cens=r(p1) if intdur<=r(p1)
		replace intdur_cens=r(p95) if intdur>=r(p95)			
		sum intdur 
		sum intdur_cens
		gen intdur_cent=intdur_cens-r(mean)
	
		rename memgap_any tempmemgap
		rename d_memgap_any d_tempmemgap
		
		label var tempmemgap "R. Temporary memory gap"
		label var d_tempmemgap "R. Temporary memory gap (ref. None)"
		label var whochg_any "R. Number of who-changes"				
		label var wherechg_any "R. Number of where-changes" 		
		label var tuactivity_n "R. Number of reported activities"
		label var activitytime "R. Average time spent on action"
		label var intdur "R. Interview duration (in minutes)"
		label var intdur_cens "R. Interview duration (in minutes, cens)"	
		label var intdur_cent "R. Interview duration (in minutes, centered)"
		label var matchedinserts "R. Matched inserts"
		
		merge 1:1 tucaseid using "$pathdta/use_atus_demographics_`today'.dta"
		tab _merge		
		keep if _merge==3
		drop _merge
		
		tab forget
		gen d_forget=forget>=1
		label var d_forget "R. ATUS enduring memory gap (indicator)"
		
		egen whowhereavg=rowmean(whochg_any wherechg_any)
		sum whowhereavg
		label var whowhereavg "R. Average Number of Who and Where Changes"
		
		recode famincdkref (2=1 "DK or REF") (3=2 "No Information"), gen(famincdkref2)
		
		sum completes
		gen completes_cent=completes-r(mean)
		label var completes_cent "I. Experience (centered)"
		
		clonevar centiverCRpr=centiverCR
		replace centiverCRpr=centiverCRpr*100
		
		order tucaseid audittrail intid iverCR centiverCR *memgap forget d_forget insufficient refusal whochg* wherechg* activitytime intdur* tuactivity_n attempts everref female age* prtage edu race ethn empl own faminc* marstat child* metro tucpsdp 
		
		save "$pathdta/ATUS_RespondentLevelData_`today'.dta", replace	
		
	log close
	exit
