program isdl, eclass
    /*
    estimates the mean of the sensitive variable from item sum double-list 
    data (ISDL) and from the direct questioning data (DQ), as well as the
    difference (IDSL-DQ)

    syntax:
        isdl Y1 Y2 G [if] [in] [, 
            smax(#) c1max(#) c2max(#) 
            mean suest sureg ml quietly 
            options ]
    
    input variables:
        Y1:    A: S+C1, B: C1,   DQ: S
        Y2:    A: C2,   B: S+C2, DQ: S
        G:     group (A=1, B=2, DQ=3)

    options:
        smax():     exclude outliers in DQ larger than smax()
        c1max():    exclude outliers in Y1 larger than smax()+c1max() (group A)
                    or larger than c1max() (group B)
        c2max():    exclude outliers in Y2 larger than c2max() (group A)
                    or larger than smax()+c2max() (group B)
        mean:       estimation using -mean-
        suest:      estimation using -regress- and -suest- 
        sureg:      estimation using -sureg-
        ml:         estimation using maximum likelihood
        quietly:    do not display intermediate results
        options:    options to be passed through to estimation command
    */
    syntax varlist(min=3 max=3 numeric) [if] [in] [, ///
        smax(numlist min=1 max=1 >0 missingok) ///
        c1max(numlist min=1 max=1 >0 missingok) ///
        c2max(numlist min=1 max=1 >0 missingok) ///
        mean suest sureg ml QUIetly * ]
    if "`smax'"=="" local smax .
    if "`c1max'"=="" local c1max .
    if "`c2max'"=="" local c2max .
    local model `mean' `suest' `sureg' `ml'
    if `:list sizeof model'>1 {
        di as err "only one of mean, suest, sureg, and ml allowed"
    }
    if "`model'"=="" local model mean
    gettoken Y1 varlist : varlist
    gettoken Y2 varlist : varlist
    gettoken G  varlist : varlist
    marksample touse
    qui replace `touse' = 0 if `G'==1 & ///
        (`Y1'>(`smax'+`c1max') | `Y2'>(`c2max')) & `touse'
    qui replace `touse' = 0 if `G'==2 & ///
        (`Y1'>(`c1max') | `Y2'>(`smax'+`c2max')) & `touse'
    qui replace `touse' = 0 if `G'==3 & `Y1'>`smax' & `touse'
    qui count if `touse'
    local N = r(N)
    qui count if `touse' & `G'==1
    local N_A = r(N)
    qui count if `touse' & `G'==2
    local N_B = r(N)
    qui count if `touse' & `G'==3
    local N_DQ = r(N)
    tempname b V
    if "`model'"!="sureg" {
        mat `b' = J(1,8,0)
        mat coln `b' = IS1 IS2 ISDL DQ IS1-IS2 IS1-DQ IS2-DQ ISDL-DQ
    }
    else {
        mat `b' = J(1,3,0)
        mat coln `b' = ISDL DQ ISDL-DQ
    }
    mat `V' = `b'
    if "`model'"!="mean" {
        tempvar A B 
        qui gen byte `A' = (`G'==1) if inlist(`G',1,2) & `touse'
        qui gen byte `B' = (`G'==2) if inlist(`G',1,2) & `touse'
    }
    if "`model'"=="mean" {
        `quietly' mean `Y1' `Y2' if `touse', over(`G', nolabel) `options'
        foreach exp in ///
            "1 _b[`Y1':1]-_b[`Y1':2]" ///
            "2 _b[`Y2':2]-_b[`Y2':1]" ///
            "3 ((_b[`Y1':1]-_b[`Y1':2]) + (_b[`Y2':2]-_b[`Y2':1]))/2" ///
            "4 _b[`Y1':3]" ///
            "5 (_b[`Y1':1]-_b[`Y1':2])-(_b[`Y2':2]-_b[`Y2':1])" ///
            "6 (_b[`Y1':1]-_b[`Y1':2])-_b[`Y1':3]" ///
            "7 (_b[`Y2':2]-_b[`Y2':1])-_b[`Y1':3]" ///
            "8 ((_b[`Y1':1]-_b[`Y1':2]) + (_b[`Y2':2]-_b[`Y2':1]))/2-_b[`Y1':3]" ///
         {
            gettoken i exp : exp
            `quietly' lincom `exp'
            mat `b'[1,`i'] = r(estimate)
            mat `V'[1,`i'] = r(se)^2
        }
    }
    else if "`model'"=="suest" {
        tempname mA mB mDQ
        `quietly' reg `Y1' `A' if `touse'
        est sto `mA'
        `quietly' reg `Y2' `B' if `touse'
        est sto `mB'
        `quietly' reg `Y1' if `touse' & `G'==3
        est sto `mDQ'
        `quietly' suest `mA' `mB' `mDQ', `options'
        foreach exp in ///
            "1 [`mA'_mean]`A'" ///
            "2 [`mB'_mean]`B'" ///
            "3 ([`mA'_mean]`A' + [`mB'_mean]`B')/2" ///
            "4 [`mDQ'_mean]_cons" ///
            "5 [`mA'_mean]`A'-[`mB'_mean]`B'" ///
            "6 [`mA'_mean]`A'-[`mDQ'_mean]_cons" ///
            "7 [`mB'_mean]`B'-[`mDQ'_mean]_cons" ///
            "8 ([`mA'_mean]`A' + [`mB'_mean]`B')/2-[`mDQ'_mean]_cons" ///
         {
            gettoken i exp : exp
            `quietly' lincom `exp'
            mat `b'[1,`i'] = r(estimate)
            mat `V'[1,`i'] = r(se)^2
        }
    }
    else if "`model'"=="sureg" {
        `quietly' nlsur (`Y1' = {S}*`A' + {C1})  (`Y2' = {S}*`B' + {C2}) ///
            , variables(`Y1' `Y2' `A' `B') `options'
        mat `b'[1,1] = el(e(b),1,1)
        mat `V'[1,1] = el(e(V),1,1)
        `quietly' reg `Y1' if `touse' & `G'==3, `options'
        mat `b'[1,2] = el(e(b),1,1)
        mat `V'[1,2] = el(e(V),1,1)
        mat `b'[1,3] = `b'[1,1]-`b'[1,2]
        mat `V'[1,3] = `V'[1,1]+`V'[1,2] // since samples are independent
    }
    else if "`model'"=="ml" {
        tempname mA mB mS mDQ
        qui reg `Y1' `A'
        local mu1 = _b[`A']
        local mu0 = _b[_cons]
        local lns = ln(e(rmse))
        `quietly' mlexp (`A'*lnnormalden(`Y1', {mu1}+{mu0}, exp({lns1})) + ///
            `B'*lnnormalden(`Y1', {mu0}, exp({lns0}))) ///
            ,  from(mu1 = `mu1' mu0 = `mu0' lns1 = `lns' lns0 = `lns') ///
            variables(`A' `B' `Y1') `options'
        local S = _b[mu1:_cons]
        local C1 = _b[mu0:_cons]
        local lns_1 = _b[lns0:_cons]
        local lns_e1 = _b[lns1:_cons]
        est sto `mA'
        qui reg `Y1' `B'
        local mu1 = _b[`B']
        local mu0 = _b[_cons]
        local lns = ln(e(rmse))
        `quietly' mlexp (`B'*lnnormalden(`Y2', {mu1}+{mu0}, exp({lns1})) + ///
            `A'*lnnormalden(`Y2', {mu0}, exp({lns0}))) ///
            , from(mu1 = `mu1' mu0 = `mu0' lns1 = `lns' lns0 = `lns') ///
            variables(`B' `A' `Y2') `options'
        local S = (`S'+_b[mu1:_cons])/2
        local C2 = _b[mu0:_cons]
        local lns_2 = _b[lns0:_cons]
        local lns_e2 = _b[lns1:_cons]
        local lns_e12 = ln(sqrt((exp(`lns_1')^2 + exp(`lns_2')^2 + ///
                        exp(`lns_e1')^2 + exp(`lns_e2')^2)/2))
        est sto `mB'
        `quietly' ml model lf isdl_lf() (S: `A' =) (C1: `Y1' =) (C2: `Y2' =) ///
            /lns_1 /lns_2 /lns_e1 /lns_e2 /lns_e12 if `touse', max ///
            init(`S' `C1' `C2' `lns_1' `lns_2' `lns_e1' `lns_e2' `lns_e12', copy) ///
            search(off) `options'
        `quietly' ml display
        est sto `mS'
        qui reg `Y1' if `touse' & `G'==3
        local b0 = _b[_cons]
        local lns = ln(e(rmse))
        `quietly' mlexp (ln(normalden(`Y1', {b0}, exp({lns})))) ///
            if `touse' & `G'==3, variables(`Y1') ///
            from(b0 = `b0' lns = `lns') `options'
        est sto `mDQ'
        `quietly' suest `mA' `mB' `mS' `mDQ'
        foreach exp in ///
            "1 [`mA'_mu1]_cons" ///
            "2 [`mB'_mu1]_cons" ///
            "3 [`mS'_S]_cons" ///
            "4 [`mDQ'_b0]_cons" ///
            "5 [`mA'_mu1]_cons-[`mB'_mu1]_cons" ///
            "6 [`mA'_mu1]_cons-[`mDQ'_b0]_cons" ///
            "7 [`mB'_mu1]_cons-[`mDQ'_b0]_cons" ///
            "8 [`mS'_S]_cons-[`mDQ'_b0]_cons" ///
         {
            gettoken i exp : exp
            `quietly' lincom `exp'
            mat `b'[1,`i'] = r(estimate)
            mat `V'[1,`i'] = r(se)^2
        }
    }
    local vce `e(vce)'
    local vcetype `e(vcetype)'
    mat `V' = diag(`V')
    eret post `b' `V', esample(`touse') obs(`N')
    eret local cmd "isdl"
    eret local model "`model'"
    eret local title "Item sum double list (`model')"
    eret local vce `vce'
    eret local vcetype `vcetype'
    eret scalar N_A = `N_A'
    eret scalar N_B = `N_B'
    eret scalar N_DQ = `N_DQ'
    _coef_table_header
    eret display
    di "Group A: N = " e(N_A) ", Group B: N = " e(N_B) ", DQ: N = " e(N_DQ)
end
program isdlreg, eclass
    /*
    maximum likelihood regression for item sum double list data; the first 
    equation in the output reports the coefficients for the sensitive item, 
    equation C1 and C2 report the coefficients for the control items. The lns_*
    terms capture the different error variances (as logarithms of standard 
    deviations).
    
    syntax:
        isdlreg Y1 Y2 G [xvars] [if] [in] [, 
            smax(#) c1max(#) c2max(#) 
            options ]
    
    input variables:
        Y1:     A: S+C1, B: C1
        Y2:     A: C2,   B: S+C2
        G:      group (A=1, B=2)
        xvars:  independent variables

    options:
        smax():     exclude outliers in DQ larger than smax()
        c1max():    exclude outliers in Y1 larger than smax()+c1max() (group A)
                    or larger than c1max() (group B)
        c2max():    exclude outliers in Y2 larger than c2max() (group A)
                    or larger than smax()+c2max() (group B)
        options:    options to be passed through to estimation command
    */
    syntax varlist(min=3 numeric) [if] [in] [, ///
        smax(numlist min=1 max=1 >0 missingok) ///
        c1max(numlist min=1 max=1 >0 missingok) ///
        c2max(numlist min=1 max=1 >0 missingok) * ]
    if "`smax'"=="" local smax .
    if "`c1max'"=="" local c1max .
    if "`c2max'"=="" local c2max .
    gettoken Y1 varlist : varlist
    gettoken Y2 varlist : varlist
    gettoken G  varlist : varlist
    marksample touse
    qui replace `touse' = 0 if `G'==1 & ///
        (`Y1'>(`smax'+`c1max') | `Y2'>(`c2max')) & `touse'
    qui replace `touse' = 0 if `G'==2 & ///
        (`Y1'>(`c1max') | `Y2'>(`smax'+`c2max')) & `touse'
    tempvar A B 
    qui gen byte `A' = (`G'==1) if `touse'
    qui gen byte `B' = (`G'==2) if `touse'
    qui count if `touse'
    local N = r(N)
    qui count if `touse' & `A'
    local N_A = r(N)
    qui count if `touse' & `B'
    local N_B = r(N)
    ml model lf isdl_lf() (S: `A' = `varlist') ///
        (C1: `Y1' = `varlist') (C2: `Y2' = `varlist') ///
        /lns_1 /lns_2 /lns_e1 /lns_e2 /lns_e12 ///
        if `touse', max `options'
    eret local title "Item sum double list regression"
    eret scalar N_A = `N_A'
    eret scalar N_B = `N_B'
    ml display
    di "Group A: N = " e(N_A) ", Group B: N = " e(N_B)
end
mata:
mata set matastrict on
void isdl_lf(transmorphic scalar M, real rowvector b, real colvector lnfj)
{   // joint ISDL estimate
    real colvector  g, y1, y2, xb, xb1, xb2
    real scalar     s_1, s_2, s_e1, s_e2, s_e12, rho_e1_2, rho_e2_1

    g       = moptimize_util_depvar(M, 1)
    y1      = moptimize_util_depvar(M, 2)
    y2      = moptimize_util_depvar(M, 3)
    xb      = moptimize_util_xb(M, b, 1)  // coefficients of S
    xb1     = moptimize_util_xb(M, b, 2)  // coefficients of C1
    xb2     = moptimize_util_xb(M, b, 3)  // coefficients of C2
    s_1     = exp(moptimize_util_xb(M, b, 4)[1])
    s_2     = exp(moptimize_util_xb(M, b, 5)[1])
    s_e1    = exp(moptimize_util_xb(M, b, 6)[1])
    s_e2    = exp(moptimize_util_xb(M, b, 7)[1])
    s_e12   = exp(moptimize_util_xb(M, b, 8)[1])
    
    rho_e1_2 = (s_e12^2 - s_e1^2 - s_2^2) / (2 * s_e1 * s_2)
    rho_e2_1 = (s_e12^2 - s_e2^2 - s_1^2) / (2 * s_e2 * s_1)
    lnfj =     g :* ln(binormalden(y1-xb-xb1, s_e1, y2-xb2, s_2, rho_e1_2)) +
        (1 :- g) :* ln(binormalden(y2-xb-xb2, s_e2, y1-xb1, s_1, rho_e2_1))
}
real colvector binormalden(real colvector x, real scalar sx, real colvector y,
    real scalar sy, real scalar r)
{
    return(1/(2*pi()*sx*sy*sqrt(1-r^2)) * exp(-1/(2*(1-r^2)) *
        (x:^2/sx^2 :+ y:^2/sy^2 :- 2*r*(x:*y)/(sx*sy))))
}
end
