###############################################################################
## Setup
###############################################################################
library(psych)
library(ltm)
library(dplyr)
library(tidyr)
library(ggplot2)
library(lavaan)

###############################################################################
# Data
###############################################################################
plik <- "ian.full.Rdata"
dir <- ""                                  # path to file
ian <- read.table(paste0(dir,plik))

###############################################################################
# Descriptive statistics
###############################################################################
n <- nrow(ian)                                        ## N
np <- nrow(ian[ian$forma=="papier",])                 ## paper participatns
ni <- nrow(ian[ian$forma=="internet",])               ## online 
nk <- nrow(ian[ian$plec=="kobieta",])                 ## women
nm <- nrow(ian[ian$plec=="mężczyzna",])               ## men
ns <- nrow(ian[ian$wykształcenie=="średnie",])        ## mid education
nw <- nrow(ian[ian$wykształcenie=="wyższe",])         ## high
no <- n - ns - nw                                     ## other education
nps <- nrow(ian[ian$forma=="papier" & ian$wykształcenie=="średnie",])
niw <- nrow(ian[ian$forma=="internet" & ian$wykształcenie=="wyższe",])

# Checking Differences
wilcox.test(ian$IAN_M~ian$forma)
wilcox.test(ian$IAN_D~ian$forma)
wilcox.test(ian$IAN_K~ian$forma)

###############################################################################
# Removing Outliners
###############################################################################
zmienne <- colnames(ian)[17:56]

D2 <- mahalanobis(ian[,zmienne], colMeans(ian[,zmienne], na.rm=T), cov(na.omit(ian[,zmienne])) )

tmp <- pchisq(D2, length(zmienne)) %>% round(.,3)
bad.boys <- which(tmp<0.001) #Tabachnick i Fidell, 2007
ian <- ian[-bad.boys,]
nrow(ian)

###############################################################################
# IRT analysis
###############################################################################
set.seed(1234)
tr_te <- sample(1:n, round(.8*n))
ian8 <- list()
ian8$trainset <- ian[tr_te,]
ian8$testset <- ian[-tr_te,]
grm.fit <- grm(ian8$trainset[,17:46], constrained=F, Hessian=T)
ian.grm <- coef.grm(grm.fit)
informacja <- NULL
for(i in 1:30){
  informacja[i] <- information(grm.fit, c(-4,4), items=i)$InfoTotal
}
tab <- data.frame(ian.grm)
tab[,6] <- round(informacja,2)
colnames(tab) <- c("b1", "b2", "b3", "b4", "a", "Inf")
print(tab)

itemy  <- list(c("IAN_1","IAN_2","IAN_5","IAN_6","IAN_9","IAN_11","IAN_16","IAN_23","IAN_30","IAN_32","IAN_33","IAN_35","IAN_39"),
               c("IAN_7","IAN_12","IAN_17","IAN_24"),
               c("IAN_10","IAN_13","IAN_15","IAN_18","IAN_20","IAN_22","IAN_27","IAN_28","IAN_29","IAN_34","IAN_36","IAN_37","IAN_40"))

itemyIRT <- list(c("IAN_9","IAN_23","IAN_30","IAN_39"), c("IAN_7","IAN_24"), c("IAN_10","IAN_29","IAN_36","IAN_40"))

###############################################################################
# RTT
###############################################################################
alpha(ian8$trainset[,itemy[[1]]])[[2]]
alpha(ian8$trainset[,itemy[[2]]])[[2]]
alpha(ian8$trainset[,itemy[[3]]])[[2]]
itemyRTT  <- list(c("IAN_2","IAN_30","IAN_33","IAN_39"),c("IAN_7","IAN_12"),c("IAN_27","IAN_28","IAN_29","IAN_36"))

###############################################################################
# EFA
###############################################################################
efa <- factanal(ian8$trainset[complete.cases(ian8$trainset),c(itemy[[1]],itemy[[2]],itemy[[3]])], factors = 3, rotation="varimax")
print(efa$loadings, cutoff=.3, sort=F)

tabela <- matrix(sapply(efa$loadings, hide3), 30,3, dimnames=dimnames(efa$loadings))
pander(tabela, caption="Loadings after Varimax rotation", round=3, justify="right")
itemyEFA  <- list(c("IAN_2","IAN_11","IAN_33","IAN_39"),c("IAN_7","IAN_17"),c("IAN_27","IAN_28","IAN_36","IAN_40"))

###############################################################################
# Main analysis
###############################################################################
wyniki <- data.frame(matrix(NA,nrow(ian8$testset),12))

nD <- 13/length(itemyIRT[[1]])
nK <- 4/length(itemyIRT[[2]])
nM <- 13/length(itemyIRT[[3]])

wyniki[,1] <- apply(ian8$testset[,itemy[[1]]],1,sum)
wyniki[,2] <- apply(ian8$testset[,itemy[[2]]],1,sum)
wyniki[,3] <- apply(ian8$testset[,itemy[[3]]],1,sum)

wyniki[,4] <- apply(ian8$testset[,itemyIRT[[1]]],1,function(x) sum(x)*nD)
wyniki[,5] <- apply(ian8$testset[,itemyIRT[[2]]],1,function(x) sum(x)*nK)
wyniki[,6] <- apply(ian8$testset[,itemyIRT[[3]]],1,function(x) sum(x)*nM)

wyniki[,7] <- apply(ian8$testset[,itemyRTT[[1]]],1,function(x) sum(x)*nD)
wyniki[,8] <- apply(ian8$testset[,itemyRTT[[2]]],1,function(x) sum(x)*nK)
wyniki[,9] <- apply(ian8$testset[,itemyRTT[[3]]],1,function(x) sum(x)*nM)

wyniki[,10] <- apply(ian8$testset[,itemyEFA[[1]]],1,function(x) sum(x)*nD)
wyniki[,11] <- apply(ian8$testset[,itemyEFA[[2]]],1,function(x) sum(x)*nK)
wyniki[,12] <- apply(ian8$testset[,itemyEFA[[3]]],1,function(x) sum(x)*nM)

colnames(wyniki) <- c("pełna.D","pełna.K","pełna.M","IRT.D","IRT.K","IRT.M","RTT.D","RTT.K","RTT.M","EFA.D","EFA.K","EFA.M")
X <- NULL
Y <- NULL
for(x in 1:3){
  for(y in 1:3){
    X <- c(X,x)
    Y <- c(Y,x+y*3)
  }
}
round(corr.test(wyniki)$r[X,Y],3)[c(1,4,7),]

###############################################################################
# Kappa coefficient
###############################################################################
library(irr)
it01 <- matrix(rep(0,3*40),40,3)
it01[it$IRT,1] <- 1
it01[it$RTT,2] <- 1
it01[it$EFA,3] <- 1
it01 <- it01[c(as.numeric(sub("IAN_","",wersje$full))),]

kappam.fleiss(it01)

###############################################################################
# Contyngency Table
###############################################################################
ian8$testset$IAN_R01 <- ian8$testset$IAN_R>=90
IAN_Rsf <- rowSums(ian8$testset[,unlist(itemy)])
density(IAN_Rsf, na.rm=T)
lines(x=density(IAN_Rsf, na.rm=T)$x*90/30.5, y=density(IAN_Rsf, na.rm=T)$y*1/2.5, pch=20, cex=.5, lty=2)
abline(v=90,lty=2)
IAN_Rsf01 <- IAN_Rsf >=30.5

###############################################################################
# Confirmatory Analysis
###############################################################################
model <- '
M =~ IAN10 + IAN6 + IAN8 + IAN3
D =~ IAN4 + IAN2 + IAN7 + IAN9
K =~ IAN5 + IAN1

IAN8 ~~ IAN10
IAN4 ~~ IAN1
IAN3 ~~ IAN1
IAN6 ~~ IAN9
IAN6 ~~ IAN3
'
fit <- cfa(model, data = ian10)
summary(fit, standardized=T)
ian.fit <- fitmeasures(fit, fit.measures = c("cfi", "tli", "agfi", "rmsea", "rmsea.ci.lower", "rmsea.ci.upper", "rmsea.pvalue", "bic"))
semPlot::semPaths(fit, what="diagram", whatLabels = "std", layout = "tree", edge.color = "black" )
modificationIndices(fit) %>% filter(mi>5) %>% arrange(desc(mi))
